/*
 * FileName:    ixTimerCtrl.c
 * Author: Intel Corporation
 * Created:     28-Aug-2002
 * Description: 
 *    Source file for timer control component.
 *
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
*/


#include <ix_ossl.h>
#include <IxOsServices.h>
#include <os_api.h>
#include <IxAssert.h>


/*
 * Put the user defined include files required
*/

#include "IxTimerCtrl.h" 
#include "IxTimerCtrl_p.h"


/*
 * Typedefs whose scope is limited to this file.
 */


typedef struct
{
    BOOL     inUse;
    unsigned id;
    BOOL isRepeating;
    ix_ossl_time_t period;
    ix_ossl_time_t expires;
    IxTimerCtrlPurpose purpose;
    IxTimerCtrlTimerCallback callback;
    void *callbackParam;
} TimerRec;

PRIVATE TimerRec timers[IX_MAX_TIMERS];
PRIVATE ix_ossl_sem_t timerRecalcSem;
PRIVATE ix_ossl_sem_t criticalSectionSem; 
PRIVATE int lastTimerId = 0;
PRIVATE unsigned highestTimerSlotUsed = 0;
PRIVATE BOOL thresholdErrorOccurred = FALSE;

/*
 * Forward declarations 
 */

PRIVATE IX_STATUS
createNewTimer(IxTimerCtrlTimerCallback func, void *param, IxTimerCtrlPurpose purpose,
	       UINT32 interval, BOOL isRepeating, unsigned *timerId);

PRIVATE TimerRec * 
evaluateTimerPriority (TimerRec * first, TimerRec * second);

PRIVATE TimerRec * 
findNextTimeout(ix_ossl_time_t now);

PRIVATE void
timerSleep(TimerRec * nextTimer, ix_ossl_time_t now);

PRIVATE void 
rescheduleTimer(TimerRec * nextTimer);

PRIVATE VOID
callTimerCallback (IxTimerCtrlTimerCallback callback, void *callbackParam);

int
timerLoop(void);

/*
 *  This function allocates a new timer.  It is used by Schedule and 
 *  ScheduleRepeating.  The value returned is the id of the timer which
 *  can be used to cancel timers.  Errors are logged but failure is not
 *  reported to the calling application.
 */

PRIVATE IX_STATUS
createNewTimer(IxTimerCtrlTimerCallback func, void *param, IxTimerCtrlPurpose purpose,
	       UINT32 interval, BOOL isRepeating, unsigned *timerId)
{
    unsigned i;
    IX_STATUS status = IX_SUCCESS;
    int osTicks; 
    ix_ossl_time_t timeVal;

    if (func == NULL)
    {
	ixOsServLog(LOG_ERROR,"client registered a NULL callback function\n",0,0,0,0,0,0); 
	return IX_TIMERCTRL_PARAM_ERROR; 
    }

    ix_ossl_tick_get(&osTicks);

    if (interval < (UINT32)(IX_CONVERT_TO_MILLISECONDS / osTicks))
    {
	ixOsServLog(LOG_ERROR,"client requested time interval (%d) finer than clock ticks\n",interval,0,0,0,0,0); 
	return IX_TIMERCTRL_PARAM_ERROR; 
    }

    *timerId = ++lastTimerId;
	
    ix_ossl_sem_take(criticalSectionSem, IX_OSSL_WAIT_FOREVER); 
    
    for (i = 0; i < IX_MAX_TIMERS; i++)
    {
        if (!timers[i].inUse)
        {
            break;
        }
    }

    if ((i >= IX_TIMER_WARNING_THRESHOLD) && (thresholdErrorOccurred == FALSE))
    {
        /*
         * This error serves as an early indication that the number of
         * available timer slots will need to be increased. This is done
         * by increasing IX_MAX_TIMERS
         */
        ixOsServLog(LOG_WARNING,"Timer threshold reached. Only %d timer slots now available\n", IX_MAX_TIMERS - i, 0,0,0,0,0);
        thresholdErrorOccurred = TRUE;
    }

    if (i == IX_MAX_TIMERS)
    {
        /* 
	 *  If you get this error, increase MAX_TIMERS above 
	 */
        ixOsServLog(LOG_ERROR, "Out of timer slots %d used - request ignored\n",i,0,0,0,0,0);
	status  = IX_TIMERCTRL_NO_FREE_TIMERS;
    }
    else
    {
        timers[i].inUse = TRUE;
        msec_To_ix_ossl_time(interval, &timeVal);
        timers[i].period = timeVal; 
      	timers[i].isRepeating = isRepeating;
	ix_ossl_time_get(&timers[i].expires);
	IX_OSSL_TIME_ADD(timers[i].expires, timers[i].period);
        timers[i].purpose = purpose;
        timers[i].callback = func;
	timers[i].callbackParam = param;
	timers[i].id = *timerId;

	if ((i) >= highestTimerSlotUsed)
	{
	    highestTimerSlotUsed = i+1;
	}
    
        ix_ossl_sem_give(timerRecalcSem);
    }

    ix_ossl_sem_give(criticalSectionSem);

    return status;
}

IX_STATUS 
ixTimerCtrlSchedule(IxTimerCtrlTimerCallback func, void *param, IxTimerCtrlPurpose purpose, UINT32 relativeTime, unsigned *timerId)
{
    return createNewTimer(func, param, purpose, relativeTime, FALSE, timerId);
}

IX_STATUS 
ixTimerCtrlScheduleRepeating(IxTimerCtrlTimerCallback func, void *param, IxTimerCtrlPurpose purpose, UINT32 interval, unsigned *timerId)
{
    return createNewTimer(func, param, purpose, interval, TRUE, timerId);
}

IX_STATUS
ixTimerCtrlCancel (unsigned id)
{
    int i;
    IX_STATUS status = IX_SUCCESS;

    ix_ossl_sem_take(criticalSectionSem, IX_OSSL_WAIT_FOREVER); 

    /* 
     *  NOTE : there is no need to get the timer callback thread to wake
     *  up and recalculate.  If this timer is the next to expire, then
     *  the timer callback thread will wake up but not find any timers to
     *  callback and will just go back to sleep again anyway.
     */

    /*
     *  NOTE2 : We cancel a timer by doing a linear search for the timer id.
     *  This is not terribly efficient but is the safest way to ensure that
     *  cancellations do not cancel the wrong timer by mistake.  Also we
     *  assume timer cancellation does not occur often.  If the timer to
     *  be cancelled is not found, an error is logged.
     */

    for (i = 0; i < (int) highestTimerSlotUsed; i++)
    {
	if (timers[i].inUse && timers[i].id == id)
	{
	    timers[i].inUse =  FALSE ;
	    break;
	}
    }

#ifndef NDEBUG
    if (i == (int) highestTimerSlotUsed)
    {
	ixOsServLog(LOG_ERROR, "Cancellation requested for invalid timer\n",0,0,0,0,0,0);
	status = IX_TIMERCTRL_PARAM_ERROR;
    }
#endif

    ix_ossl_sem_give(criticalSectionSem);

    return status;
}

PRIVATE TimerRec * 
evaluateTimerPriority (TimerRec * first, TimerRec * second)
{
    IX_ASSERT (first != NULL || second != NULL);

    if (first == NULL)
    {
	return second;
    }

    if (second == NULL)
    {
	return first;
    }

    /*
     *  For now we just compare the values of the purpose with lower
     *  values getting higher priority.
     *
     *  If someone needs to change priorities of different purposes without
     *  modifying the order of the enums, then more code will be required
     *  here.
     */

    if (first->purpose <= second->purpose)
    {
	return first;
    }

    return second;
}

/*
 *  Find the next timer about to go off.  Do this by starting with a time infinitely
 *  far in the future and successively finding better timers (ones that go off 
 *  sooner).
 *
 *  If a timer is found that has already expired, the flag bestTimerExpired is
 *  set.  If another timer is found that has also expired, their respective 
 *  priorities are compared using the function evaluateTimerPriority()
 */

PRIVATE TimerRec * 
findNextTimeout(ix_ossl_time_t now)
{
    ix_ossl_time_t timeoutAt = {ULONG_MAX, LONG_MAX};
    unsigned i;
    TimerRec * bestTimer = NULL;
    BOOL bestTimerExpired = FALSE;
    BOOL thisTimerExpired = FALSE;

    for (i = 0 ; i < highestTimerSlotUsed; i++)
    {
	if (timers[i].inUse)
	{
            thisTimerExpired = FALSE;

	    /* Check if this timer has expired,
	       i.e. 'now' must be greater than timers[i].expired */

	    if(!IX_OSSL_TIME_GT(timers[i].expires, now))
	    {
		thisTimerExpired = TRUE;
	    }

	    /* If more than one timer has expired, determine
	       which callback to call first based on a priority scheme
	       i.e. the bestTimer*/

	    if((bestTimerExpired && thisTimerExpired) ||
	       IX_OSSL_TIME_EQ(timers[i].expires, timeoutAt))
	    {
		bestTimer = evaluateTimerPriority(bestTimer, &timers[i]);
		timeoutAt = bestTimer->expires;
	    }
	    else if(IX_OSSL_TIME_LT(timers[i].expires, timeoutAt))
	    {
		bestTimer = &timers[i];
		timeoutAt = timers[i].expires;
	    }

            /*
	     *  bestTimer can not be NULL here because any timer will
	     *  have a shorter timeout than the default.
	     */
	    if (!IX_OSSL_TIME_GT(bestTimer->expires, now))
	    {
		bestTimerExpired = TRUE;
	    }
	}
    }
    return bestTimer;
}

PRIVATE void
timerSleep(TimerRec * nextTimer, ix_ossl_time_t now)
{
    unsigned ticks;
    ix_ossl_time_t temp;

    if (nextTimer == NULL)
    {
	ticks = IX_OSSL_WAIT_FOREVER;
    }
    else
    {
	temp.tv_sec = nextTimer->expires.tv_sec;
	temp.tv_nsec = nextTimer->expires.tv_nsec;
	
	IX_OSSL_TIME_SUB(temp, now);


	IX_OSSL_TIME_CONVERT_TO_TICK( ticks, temp);

	/* We should sleep but the period is less than a tick
	 * away, rounding up. 
	 */
	if (ticks == 0)
	{
	    ticks = 1;
	}
    }

    ix_ossl_sem_take(timerRecalcSem, ticks);
   
}

PRIVATE void 
rescheduleTimer(TimerRec * nextTimer)
{
    if (nextTimer->isRepeating)
    {
	IX_OSSL_TIME_ADD(nextTimer->expires, nextTimer->period);
    }
    else
    {
	nextTimer->inUse = FALSE ;
    }
}

PRIVATE UINT32 timerCallbacksCalled = 0;

/*  
 *  Maximum time spent in a call-back is defined as 100 millisec.  (one 
 *  tenth of a second).
 */


PRIVATE VOID
callTimerCallback (IxTimerCtrlTimerCallback callback, void *callbackParam)
{
    ix_ossl_time_t callBackBegin, callBackFinish;
    static BOOL errorReported = FALSE;

    timerCallbacksCalled++;
    ix_ossl_time_get(&callBackBegin);

    callback(callbackParam);

    ix_ossl_time_get(&callBackFinish);
    
    IX_OSSL_TIME_SUB(callBackFinish,callBackBegin);

    if ( (callBackFinish.tv_nsec > IX_MAX_TIMER_CALLBACK_TIME) || (callBackFinish.tv_sec > 0) ) 
    {
	if (!errorReported)
	{
	    ixOsServLog(LOG_WARNING,"Slow timer callback - %d Secs, %d nSecs \n",callBackFinish.tv_sec,callBackFinish.tv_nsec,0,0,0,0);
	    errorReported = TRUE;
	}
    }
}

int
timerLoop(void)
{
    ix_ossl_time_t now;
    IxTimerCtrlTimerCallback callback = NULL;
    void *callbackParam = NULL;
    TimerRec * nextTimer;

    while (1)
    {
        /*
         *  This loop catches all cases in a simple way.  If multiple
         *  timers expire together, then lowest will be <=0 until all
	 *  have been processed and the queue get won't get invoked.
         */

	ix_ossl_sem_take(criticalSectionSem, IX_OSSL_WAIT_FOREVER); 
	ix_ossl_time_get(&now);
	nextTimer = findNextTimeout(now);
	
	if ((nextTimer == NULL) || IX_OSSL_TIME_GT(nextTimer->expires, now))
	{
	    callback = NULL;
	}
	else
	{
	    rescheduleTimer(nextTimer);
	    callback = nextTimer->callback;
	    callbackParam = nextTimer->callbackParam;
	}

	ix_ossl_sem_give(criticalSectionSem);

        if (callback != NULL)
        {
	    callTimerCallback(callback, callbackParam);
        }
        else
        {
	    timerSleep(nextTimer, now);
        }
    }
}

void
ixTimerCtrlShow(void)
{
    unsigned i = 0;
    unsigned count = 0;

    ixOsServLog(LOG_USER, "Timers\n",0,0,0,0,0,0);

    for (i = 0; i < highestTimerSlotUsed; i++)
    {
	if (timers[i].inUse == TRUE)
	{
	    ixOsServLog(LOG_USER, "id=%d, repeat=%d, purpose=%d\n",timers[i].id, timers[i].isRepeating, timers[i].purpose,0,0,0);
	    count++;
	}
    }
    ixOsServLog(LOG_USER, "total=%d\n", count,0,0,0,0,0);
    ixOsServLog(LOG_USER, "num called=%u\n", timerCallbacksCalled,0,0,0,0,0); 
    ixOsServLog(LOG_USER, "Timer threshold reached : %d\n",thresholdErrorOccurred, 0,0,0,0,0);

}



IX_STATUS
ixTimerCtrlInit(void)
{
    ix_ossl_thread_t taskId;
    ix_error osError;

    /* 
     *  Critical semaphore to protect the timer tables 
     */
    

    osError = ix_ossl_sem_init(IX_OSSL_SEM_AVAILABLE, &criticalSectionSem);
    if (osError != IX_OSSL_ERROR_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Error creating critical section semaphore\n",0,0,0,0,0,0);
	return IX_FAIL;
    }
    IX_ASSERT (osError == IX_OSSL_ERROR_SUCCESS);

    osError = ix_ossl_sem_init(IX_OSSL_SEM_UNAVAILABLE, &timerRecalcSem);

    if (osError != IX_OSSL_ERROR_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Error creating timer recalc semaphore\n",0,0,0,0,0,0);
	return IX_FAIL;
    }
    IX_ASSERT (osError == IX_OSSL_ERROR_SUCCESS);


    osError = ix_ossl_thread_create((ix_ossl_thread_entry_point_t)timerLoop,NULL,&taskId);
    if (osError != IX_OSSL_ERROR_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Error creating timerLoop task\n",0,0,0,0,0,0);
	return IX_FAIL;
    }
    IX_ASSERT (osError == IX_OSSL_ERROR_SUCCESS);

    return IX_SUCCESS;
}

PRIVATE void 
msec_To_ix_ossl_time(UINT32 ms, ix_ossl_time_t *timeVal)
{
    IX_OSSL_TIME_ZERO((*timeVal));

    if( ms > 0 )
    {
	timeVal->tv_sec = ms / 1000;
	timeVal->tv_nsec = (ms % 1000) * 1000000;
    }
}
