/**
 * @file    IxTimersCodelet.h
 *
 * @date    1-Apr-2003
 *
 * @brief   Header file for the IXP425 hardware timers
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 *
 * 
 *
 */

/**
 * @defgroup Codelets IXP425 Codelets
 *
 * @brief IXP425 demo codelets
 *
 * @{
 */

/**
 * @ingroup Codelets
 *
 * @defgroup IxTimerAPI IXP425 Timers (IxTimersCodelet) API
 *
 * @brief IXP425 Timer component API
 *
 * These utilities provide support for enabling, triggering and 
 * disabling timers
 *
 * @{
 */

#ifndef IXTIMERSCODELET_H_
#define IXTIMERSCODELET_H_

#include "IxTypes.h"


/**
*
* @ingroup IxTimerAPI
*
* @enum IxTimerId
*
* @brief Hardware timers
*
*/
typedef enum
{
    IX_TIMER_1,   /**< General Purpose Timer 1 */
    IX_TIMER_2,   /**< General Purpose Timer 2 */
    IX_TIMER_WDOG,/**< Watchdog Timer */
    IX_TIMER_TS,  /**< Timestamp Timer */
    IX_TIMER_PMU, /**< XScale core PMU timer */
    IX_TIMER_MAX  /**< Delimiter for error checking */
} IxTimerId;

/**
 * @ingroup IxTimerAPI
 *
 * @typedef IxTimerIsr
 *
 * @brief Timer callback prototype
 */
typedef void (*IxTimerIsr)(void *arg);

/**
 *
 * @ingroup IxTimerAPI
 *
 * @fn ixTimerInit (BOOL recoverFromLostStatus)
 *
 * @brief Initialise the Timer API
 *
 * This function must be called before and other IxTimer function. It
 * sets up internal data structures.
 *
 * @param recoverFromLostStatus (in) set to true to enable
 *              Lost Status workaround
 *
 * @return @li IX_SUCCESS, the IxTimer API successfully initialised
 * @return @li IX_FAIL, failed to initialize the IxTimer API
 *
 */
PUBLIC IX_STATUS 
ixTimerInit (BOOL recoverFromLostStatus);


/**
 *
 * @ingroup IxTimerAPI
 *
 * @fn ixTimerBind (IxTimerId timer, IxTimerIsr isr, void *arg)
 *
 * @brief Initialise a Timer 
 *
 * This function is called to initialise one of the timers
 *
 * @param timer (in) the timer to initialise
 * @param isr (in) the callback to register
 * @param arg (in) the parameter to be passed to the callback
 *
 * @return @li IX_SUCCESS, the timer successfully initialised
 * @return @li IX_FAIL, failed to initialize the timer
 *
 */
PUBLIC IX_STATUS
ixTimerBind (IxTimerId timer,
	     IxTimerIsr isr, 
	     void *arg);


/**
 *
 * @ingroup IxTimerAPI
 *
 * @fn ixTimerEnable (IxTimerId timer, UINT32 downCounter, BOOL oneShot)
 *
 * @brief Enable a Timer 
 *
 * This function is called to enable a timer
 *
 * @param timer (in) the timer to enable
 * @param downCounter (in) the number of cycles between interrupts.
 *        downCounter should be a multiple of 4 for @a IX_TIMER_1
 *        and @a IX_TIMER_2 timers.
 *        This counter is in pClock cycles (peripheral cycles)
 *        except for the @a IX_TIMER_PMU where the unit is 
 *        in xClock cycles (processor frequency)
 * @param oneShot (in) the type of timer to trigger. When set to
 *        TRUE, the interrupt will fire once when the downCounter
 *        reaches 0. When set to FALSE, the interrupts will fire
 *        every @p downCounter bus cycles. FALSE is not supported 
 *        for @a IX_TIMER_WDOG and @a IX_TIMER_TS timers.
 *
 * @return @li IX_SUCCESS, the timer successfully enabled
 * @return @li IX_FAIL, failed to enable the timer
 *
 * @note This function can be used from an interrupt context
 *
 */
PUBLIC IX_STATUS 
ixTimerEnable (IxTimerId timer,
	       UINT32 downCounter,
	       BOOL oneShot);

/**
 *
 * @ingroup IxTimerAPI
 *
 * @fn ixTimerDisable (IxTimerId timer)
 *
 * @brief Disable a Timer 
 *
 * This function is called to disable a timer
 *
 * @param timer (in) the timer to disable
 *
 * @return @li IX_SUCCESS, the timer successfully disabled
 * @return @li IX_FAIL, failed to disable the timer
 *
 * @note This function can be used from an interrupt context
 *
 */
PUBLIC IX_STATUS 
ixTimerDisable (IxTimerId timer);

/**
 *
 * @ingroup IxTimerAPI
 *
 * @fn ixTimerDownCounterGet (IxTimerId timer, UINT32 *downCounter)
 *
 * @brief Get the remaining time of a timer 
 *
 * This function is called to get the remaining time before the next 
 * interrupt triggers
 *
 * If this function is used after a timer expires, the 
 * result is a negative value. Its unsigned representation is a number 
 * close to 0xfffff.....
 *
 * @note Watchdog timer stops when expiring. Then the counter retrived
 * by this function is always 0.
 *
 * @param timer (in) the timer to query
 * @param downCounter (out) the timer time left 
 *
 * @return @li IX_SUCCESS, the timer successfully queried
 * @return @li IX_FAIL, failed to query the timer
 *
 * @note This function can be used from an interrupt context
 *
 */
PUBLIC IX_STATUS 
ixTimerDownCounterGet (IxTimerId timer, UINT32 *downCounter);

/**
 *
 * @ingroup IxTimerAPI
 *
 * @fn ixTimerUnbind (IxTimerId timer)
 *
 * @brief Unregister a Timer 
 *
 * This function is called to unregister a timer
 *
 * @param timer (in) the timer to unregister
 *
 * @return @li IX_SUCCESS, the timer successfully removed
 * @return @li IX_FAIL, failed to remove the timer
 *
 */
PUBLIC IX_STATUS 
ixTimerUnbind (IxTimerId timer);

/**
 *
 * @ingroup IxTimerAPI
 *
 * @fn ixTimerShow
 *
 * @brief Show internal counters
 *
 * This function is called to show the internal counters
 * icremented when Lost Status workaround is needed.
 *
 * @param none
 *
 * @return @li void
 *
 */
PUBLIC void 
ixTimerShow(void);


#endif /* IXTIMERSCODELET_H_ */

/**
 * @} defgroup IxTimerAPI
 */

