/**
 * @file IxHssAccCodelet.h
 *
 * @date 26 Mar 2002
 *
 * @brief This file contains the interface for the HSS Access Codelet.
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 *
 */
#ifndef IXHSSACCCODELET_H
#define IXHSSACCCODELET_H

/**
 * @defgroup Codelets IXP425 Codelets
 *
 * @brief IXP425 Codelets
 *
 * @{
 */

/**
 * @ingroup Codelets
 *
 * @defgroup IxHssAccCodelet IXP425 HSS Access Codelet (IxHssAccCodelet) API
 *
 * @brief IXP425 HSS Access Codelet API. The interface for the HSS Access Codelet.
 *
 * This module contains the implementation of the HSS Access Codelet.
 * <P>
 * The following top-level operation is supported:
 * <UL>
 *   <LI>Test Packetised and Channelised Services, with the Codelet acting
 *       as data source/sink and HSS as loopback.
 *   The Codelet will transmit data and verify that data received is the
 * same as that transmitted. Codelet runs for IX_HSS_CODELET_DURATION_IN_MS ms. 
 *   <LI>There are four clients of Packetised service per HSS port - 1 client
 *	 per E1/T1 trunk. Client 0 and 2 are running in Packetised HDLC mode while 
 *	 client 1 and 3 in Packetised RAW mode.
 * </UL>
 *
 * <b> Assumptions </b><br>
 * In Channelised service, the codelet transmits traffic continuously. When
 * the codelet runs up to IX_HSS_CODELET_DURATION_IN_MS ms, Tx counter is bigger 
 * than Rx counter. This is due to the fact that traffics submitted to NPE (i.e. 
 * Tx counter has been increased) are not transmitted out by NPE when HSS service
 * is disabled. These traffics will be dropped and not loopbacked at HSS (Hence, 
 * Rx counter not increased).<br>
 *
 * In Packetised-raw mode service (client 1 and 3), Rx counter will be bigger than 
 * Tx counter because in this service, idle packets are received by XScale and
 * causes Rx counter to be bigger than Tx counter. As for packetised-HDLC service, 
 * idle packets are handled in HDLC coprocessor and not passed to XScale (Hence,
 * Rx counter not increased).     
 *
 * <b> Limitations </b><br>
 * When executing Packetised service on both HSS ports of 266MHz IXP421 
 * simultaneously, receive traffic verification should fail on client 3 (i.e. 
 * Packetised-raw mode) of HSS port 1. The reason why this issue occured is IXP421
 * does not have enough CPU resources to perform intensive packet verification 
 * tasks. However, this does not imply that the same issue will hit customer 
 * applications because actual applications will not do any packet verifications 
 * in the way that the codelet does. This issue is not seen on a 533MHz IXP425.
 *
 * <b> VxWorks User Guide </b><br>
 * ixHssAccCodeletMain() function is used as a single point of execution for 
 * HssAcc Codelet.  
 *
 * <pre>
 *  <i> Usage :
 *      >ixHssAccCodeletMain (operationType, portMode, verifyMode)
 *      Where operationType:
 *           1 = Packetised Service Only.
 *           2 = Channelised Service Only.
 *           3 = Packetised Service and Channelised Services.
 *
 *      Where portMode:
 *           1 = HSS Port 0 Only.
 *           2 = HSS Port 1 Only.
 *           3 = HSS Port 0 and 1.
 *
 *      Where verifyMode:
 *           1 = codelet verifies traffic received in hardware loopback mode.
 *           2 = codelet does not verify traffic received in hardware loopback mode.
 * 
 * </i>
 * </pre>
 *
 * <b> Linux User Guide </b><br>
 * The idea of using the ixHssAccCodeletMain() as a single point of execution for
 * HssAcc codelet. The operation selected will be executed when user issue 'insmod' 
 * in command prompt.
 *
 * <pre>
 * <i>  Usage :
 *      >insmod ixp400_codelets_hssAcc.o operationType=(a) portMode=(b) verifyMode=(c)
 *      Where a:
 *           1 = Packetised Service Only.
 *           2 = Channelised Service Only.
 *           3 = Packetised Service and Channelised Services.
 *        
 *      Where b:
 *           1 = HSS Port 0 Only.
 *           2 = HSS Port 1 Only.
 *           3 = HSS Port 0 and 1.
 *
 *      Where c:
 *           1 = codelet verifies traffic received in hardware loopback mode.
 *           2 = codelet does not verify traffic received in hardware loopback mode.
 *
 * </i>
 * </pre>
 * <P>
 * <B>Buffer Management</B>
 * <P>
 * The packetised service uses mbuf buffers to store data, and chains mbufs
 * together to form large packets.  In the transmit direction, mbufs are
 * allocated from a pool on transmit, and returned to the pool on tranmsit
 * done.  For receive, mbufs are allocated from a pool when supplying
 * buffers to the free queue, and returned to the pool on receive.
 * <P>
 * The channelised service operates quite differently.  As voice data is
 * very sensitive to latency using mbufs for transferral of the data
 * between XScale and NPE is not very appropriate.  Instead, circular
 * buffers are used whereby the NPE reads data from a block of SDRAM that
 * the XScale writes to, and writes data to a block of SDRAM that the
 * XScale reads from.  On receive, the NPE writes directly into a circular
 * buffer that the XScale subsequently reads the data from.  Each channel
 * has its own circular buffer, and all these buffers are stored
 * contiguously.  On transmit, the NPE takes a circular list of pointers
 * from the XScale and transmits the data referenced by these pointers.
 * Each list of pointers contains a pointer for each channel, and the
 * circular list of pointers contains multiple lists stored contiguously.
 * This is to allow the XScale to transmit voice samples without having to
 * copy data, as only the pointer to the data blocks needs to be written to
 * SDRAM.  The NPE lets the XScale know, in the form of Tx and Rx offsets,
 * where in the blocks of SDRAM it is currently reading from and writing
 * to.  This enables the XScale to co-ordinate its reading and writing
 * activities to maintain the data flow.  The Tx offset lets the XScale
 * know the list offset into the Tx circular pointer list that the NPE will
 * next use to transmit.  The Rx offset lets the XScale know the byte
 * offset into each channel's Rx circular buffer that the NPE will next
 * receive data into.
 * <P>
 * <B>Caching</B>
 * <P>
 * To improve system performance, caching may be enabled for both the
 * channelised and packetised buffers.  To allow for this, buffers need to
 * be flushed before transmit, and invalidated after receive.  Flushing the
 * buffers before transmit ensures the NPE reads and transmits the correct
 * data.  Invalidating the buffers after receive ensures the XScale reads
 * and processes the correct data.  In the case of the Codelet, all data is
 * flushed and invalidated as every byte is being written to on transmit
 * and every byte is verified on receive.  In a real application flushing
 * or invalidating all the data may not be necessary, only the data that
 * the application has written before transmit or will read after receive.
 * Note, regarding the packetised service, the IxHssAcc component itself
 * takes care of flushing and invalidating mbuf headers.  The application
 * needs only to concern itself with the mbuf data.
 * <P>
 * <B>Data Verification Strategy</B>
 * <P>
 * For both the packetised and channelised service a changing pattern will
 * be transmitted.  When the HSS co-processor is performing a loopback the
 * data received is expected to be the same as that transmitted.  The data
 * transmitted carries a byte pattern that begins at a known value and is
 * incremented for each byte.  An independent byte pattern is transmitted
 * for each channel of the channelised service, and also for each port of
 * the packetised service.  When data is received it is expected to match
 * the pattern that was transmitted.  For the channelised service the first
 * non-idle byte received is expected to be the beginning of the byte
 * pattern.  For the packetised service, RAW mode clients may receive idle
 * data so this is detected and ignored.  Only non-idle data is verified.
 * <P>
 * <B>56Kbps Packetised HDLC feature</B>
 * <P>
 * This feature is demonstrated in one of the packetised HDLC clients (i.e. 
 * client 2). The CAS bit is configured to be in the least signicant bit 
 * (LSB) position with bit polarity '1'. Bit inversion is also enabled on 
 * this client as well. The data verification strategy remains the same as
 * other packetised clients.
 * 
 * @{
 */

#include "IxTypes.h"
#include "IxHssAccCodelet_p.h"

/*
 * #defines for function return types, etc.
 */

/**
 * @ingroup IxHssAccCodelet                
 *
 * @def IX_HSSACC_CODELET_DURATIO_IN_MS
 *
 * @brief Duration (in ms) of codelet.
 *  
 */
#define IX_HSSACC_CODELET_DURATION_IN_MS 10000

/*
 * Prototypes for interface functions.
 */

/** 
 * @ingroup IxHssAccCodelet
 *
 * @fn ixHssAccCodeletMain(IxHssAccCodeletOperation operationType,
 *                         IxHssAccCodeletPortMode portMode,
 *                         IxHssAccCodeletVerifyMode verifyMode)
 *
 * @brief This function is used as a single point of execution for HssAcc codelet.
 *
 * @param "IxHssAccCodeletOperation [in] operationType" - The type of operation 
 *        to be executed. Refer to the descriptions above.
 * @param "IxHssAccCodeletPortMode [in] portMode" - portMode supported.
 *        Refer to the descriptions above.
 * @param " IxHssAccCodeletVerifyMode [in] verifyMode" - verify mode supported.
 *        Refer to the description above.
 *
 * @return
 * - IX_SUCCESS : If operation selected is successfully setup
 * - IX_FAIL    : If operation selected fails to be setup.
 */
PUBLIC IX_STATUS
ixHssAccCodeletMain(IxHssAccCodeletOperation operationType,
                    IxHssAccCodeletPortMode portMode,
                    IxHssAccCodeletVerifyMode verifyMode);

#endif /* IXHSSACCCODELET_H */

/** @} defgroup IxHssAccCodelet*/

/** @} defgroup Codelets*/


