/*
 * FileName:    IxEthAal5AppUtils.c
 *
 * Created:     15 May 2002
 *
 * Description: Codelet utils for EthAal5 Application
 *
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 */

/*
 * System defined include files
 */
#include <string.h>

/*
 * User defined include files
 */
#include "IxTypes.h"
#include "IxAtmm.h"
#include "IxAtmTypes.h"
#include "IxAtmSch.h"
#include "IxAtmdAcc.h"
#include "IxOsServices.h"
#include "IxOsServicesMemMap.h"
#include "IxNpeMh.h"
#include "IxAssert.h"
#include "IxNpeMicrocode.h"
#include "ixp425.h"
#include "IxQMgr.h"
#include "IxEthAal5App_p.h"

/*
 * #defines and macros
 */
#define NO_UTOPIA_BUILD_ID_        1

#define BSR_CTRL1 0x2

#define EXPBUS_REGS_LOCATION    IX_OSSERV_EXP_BUS_PHYS_BASE 

/* EXPBus location in memory map */
#define EXP_MEMMAP_BOOTMODE     IX_OSSERV_EXP_BUS_BOOT_PHYS_BASE
#define EXP_MEMMAP_NORMALMODE   IX_OSSERV_EXP_BUS_PHYS_BASE 

/* EXPBus register address definitions */
#define EXP_TIMING_CS5  (EXPBUS_REGS_LOCATION + 0x14)

#define EXPBUS_STIMULUS_CS      5       /* CS line index*/
#define EXP_TIMING_CS_STIMULUS  EXP_TIMING_CS5 /* CS cfg register address 
						  dedicated to stimulus */

/* EXPBus configuration */
#define EXP_CNFG0       (EXPBUS_REGS_LOCATION + 0x20) /* Configuration 0 */
#define EXP_CNFG1       (EXPBUS_REGS_LOCATION + 0x24) /* Configuration 1 */
#define EXP_CNFG2       (EXPBUS_REGS_LOCATION + 0x28) /* Configuration 2 */

/* EXPBus EXP_TIMING_CSx bit definitions */
#define EXP_TIMING_CS_EN_MASK   0x80000000  /* Bit 31 - Chip select enabled 
					       when set to 1 */
#define EXP_TIMING_CS_T1_MASK   0x30000000  /* Bit 29:28 - T1 Address timing 
					       (0 - normal timing) */
#define EXP_TIMING_CS_T2_MASK   0x0C000000  /* Bit 27:26 - T2 setup phase 
					       timing (0 - timing) */
#define EXP_TIMING_CS_T3_MASK   0x03C00000  /* Bit 25:22 - T3 strobe phase 
					       timing (0 - normal timing) */
#define EXP_TIMING_CS_T4_MASK   0x00300000  /* Bit 21:20 - T4 hold phase 
					       timing (0 - normal timing) */
#define EXP_TIMING_CS_T5_MASK   0x000F0000  /* Bit 19:16 - T5 recovery phase 
					       timing (0 - normal timing) */
#define EXP_TIMING_CS_CNFG_MASK 0x00003C00  /* Bit 13:10 - device mem size = 
					       2^(9+CNFG) : 0 - 2^9 = 512 
					       bytes */
#define EXP_TIMING_CS_WR_MASK   0x00000002  /* Bit 1 - WR_EN - enables writes 
					       when set to 1 */

/* configuration used by stimulus */
#define EXP_TIMING_CFG_STIMULUS  ( EXP_TIMING_CS_EN_MASK | EXP_TIMING_CS_T1_MASK |EXP_TIMING_CS_T2_MASK | EXP_TIMING_CS_T3_MASK | EXP_TIMING_CS_T4_MASK | EXP_TIMING_CS_T5_MASK | EXP_TIMING_CS_CNFG_MASK | EXP_TIMING_CS_WR_MASK )

/* EXPBus EXP_CNFG0 bit definitions Location of EXPBus in memory space: 
 * 1 - 0x0, 0 - 0x50000000*/
#define EXP_CNFG0_MEMMAP_MASK   0x80000000


#define DISCONNECT_RETRY_DELAY     (25)
#define DISCONNECT_RETRY_COUNT     (200)
#define UTOPIA_PORT_RATE           (53 * 1000000)


/*
 * Typedefs
 */
typedef struct
{
    BOOL inUse;
    IxAtmConnId connId;
    IxAtmLogicalPort port;
} ChannelInfo;

/*
 * Variable declarations global to this file. Externs are followed by
 * statics.
 */
static ChannelInfo rxChannelInfo[IX_ATM_MAX_NUM_VC];
static ChannelInfo txChannelInfo[IX_ATM_MAX_NUM_VC];

static IxAtmSchedulerVcId rxVcId[IX_ATM_MAX_NUM_VC];
static IxAtmNpeRxVcId rxNpeVcIdTable[IX_ATM_MAX_NUM_VC];

static IxAtmSchedulerVcId txVcId[IX_ATM_MAX_NUM_VC];

static IxAtmLogicalPort rxPort[IX_ATM_MAX_NUM_VC];
static IxAtmLogicalPort txPort[IX_ATM_MAX_NUM_VC];

/* function prototype */
PRIVATE void writeStimulusReg(unsigned uRegOffset, UINT16 uregValue);
PRIVATE UINT16 readStimulusReg(unsigned uRegOffset);
PRIVATE void registerWrite(volatile unsigned *reg, unsigned value);
PRIVATE void fpgaStimulusEnable(void);
PRIVATE IX_STATUS rxFreeChannelGet (unsigned int *channelId);
PRIVATE IX_STATUS txFreeChannelGet (unsigned int *channelId);
PRIVATE IX_STATUS rxChannelFind (IxAtmConnId connId, unsigned int *channelId);
PRIVATE IX_STATUS txChannelFind (IxAtmConnId connId, unsigned int *channelId);
PRIVATE IX_STATUS txVcDisconnect(IxAtmConnId txConnId);
PRIVATE IX_STATUS rxVcDisconnect(IxAtmConnId rxConnId);

/*
 * Function definitions.
 */


/* Write to stimulus register. uRegOffset is an offset from start of EXPBus 
 * mem. Function properly recognizes current EXPBus mapping */
PRIVATE void    
writeStimulusReg(unsigned uRegOffset, UINT16 uregValue)
{
    UINT32  uCurrentEXPBusLocation;
        
    /* Figure out where EXPBus address space is mapped (from AHB point if
     * view) */
    if( *( (UINT32*) (EXP_CNFG0) ) & EXP_CNFG0_MEMMAP_MASK) 
        uCurrentEXPBusLocation = EXP_MEMMAP_BOOTMODE;
    else
        uCurrentEXPBusLocation = EXP_MEMMAP_NORMALMODE;
        
    /* Write value to EXPBus register (CS, on which stimuls resides is coded 
     * into bit 26:24) */
    *( (UINT16*) (uCurrentEXPBusLocation + uRegOffset + (EXPBUS_STIMULUS_CS<<24) ) ) = uregValue;
}

/* Read stimulus register. uRegOffset is an offset from start of EXPBus mem. 
 * Function properly recognizes current EXPBus mapping */
PRIVATE UINT16  
readStimulusReg(unsigned uRegOffset)
{
    UINT32  uCurrentEXPBusLocation;
            
    /* Figure out where EXPBus address space is mapped 
     * (from AHB point if view) */
    if( *( (UINT32*) (EXP_CNFG0) ) & EXP_CNFG0_MEMMAP_MASK) 
        uCurrentEXPBusLocation = EXP_MEMMAP_BOOTMODE;
    else
        uCurrentEXPBusLocation = EXP_MEMMAP_NORMALMODE;
        
    /* Read value from EXPBus register (CS, on which stimuls resides is coded 
     * into bit 26:24) */
    return *( (UINT16*) (uCurrentEXPBusLocation + uRegOffset + (EXPBUS_STIMULUS_CS<<24) ) );
}

PRIVATE void 
registerWrite(volatile unsigned *reg, unsigned value)
{
  *reg = value;
}

/* Function configures EXPBus to enable INTEL/16-bit mode on CS5 used for 
 * configuration of stimulus */
PRIVATE void 
fpgaStimulusEnable(void)
{
    /* configure CS, on which resides stimulus hardware to support 
     * INTEL/16-bit, non-multiplexed */
    registerWrite( (unsigned*) (EXP_TIMING_CS_STIMULUS), EXP_TIMING_CFG_STIMULUS );
}

PRIVATE IX_STATUS
rxFreeChannelGet (unsigned int *channelId)
{
    int i;
    for (i = 0; i < IX_ATM_MAX_NUM_VC; i ++)
    {
	if (!rxChannelInfo[i].inUse)
	{	    
	    *channelId = i;
	    rxChannelInfo[i].inUse = TRUE;
	    return IX_SUCCESS;
	}
    }
    return IX_FAIL;
}

PRIVATE IX_STATUS
txFreeChannelGet (unsigned int *channelId)
{
    int i;
    for (i = 0; i < IX_ATM_MAX_NUM_VC; i ++)
    {
	if (!txChannelInfo[i].inUse)
	{	    
	    *channelId = i;
	    txChannelInfo[i].inUse = TRUE;
	    return IX_SUCCESS;
	}
    }
    return IX_FAIL;
}

PRIVATE IX_STATUS
rxChannelFind (IxAtmConnId connId, unsigned int *channelId)
{
    int i;
    for (i = 0; i < IX_ATM_MAX_NUM_VC; i ++)
    {
	if ((rxChannelInfo[i].inUse) && (rxChannelInfo[i].connId == connId))
	{	    
	    *channelId = i;
	    return IX_SUCCESS;
	}
    }
    return IX_FAIL;
}

PRIVATE IX_STATUS
txChannelFind (IxAtmConnId connId, unsigned int *channelId)
{
    int i;
    for (i = 0; i < IX_ATM_MAX_NUM_VC; i ++)
    {
	if ((txChannelInfo[i].inUse) && (txChannelInfo[i].connId == connId))
	{	    
	    *channelId = i;
	    return IX_SUCCESS;
	}
    }
    return IX_FAIL;
}

PRIVATE IX_STATUS
txVcDisconnect(IxAtmConnId txConnId)
{
    IX_STATUS retval;
    unsigned int retryCount;

    retryCount = 0;
    do
    {
	retval = ixAtmdAccTxVcTryDisconnect(txConnId);
	if (retval == IX_ATMDACC_RESOURCES_STILL_ALLOCATED)
	{
	    ixOsServTaskSleep(DISCONNECT_RETRY_DELAY);
	}
    }
    while ((retryCount++ < DISCONNECT_RETRY_COUNT)
	   && (retval == IX_ATMDACC_RESOURCES_STILL_ALLOCATED));

    if (retval != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "tx disconnect failed (%d)\n",
		    retval, 0, 0, 0, 0, 0);
	return retval;
    }

    return IX_SUCCESS;
}

PRIVATE IX_STATUS
rxVcDisconnect(IxAtmConnId rxConnId)
{
    IX_STATUS retval;
    unsigned int retryCount;

    retval =  ixAtmdAccRxVcDisable(rxConnId);
    if (retval != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "rx disable failed (%d)\n", retval, 0, 0, 0, 0, 0);
	return retval;
    }

    retryCount = 0;
    do
    {
	retval = ixAtmdAccRxVcTryDisconnect(rxConnId);
	if (retval == IX_ATMDACC_RESOURCES_STILL_ALLOCATED)
	{
	    ixOsServTaskSleep(DISCONNECT_RETRY_DELAY);
	}
    }
    while ((retryCount++ < DISCONNECT_RETRY_COUNT)
	   && (retval == IX_ATMDACC_RESOURCES_STILL_ALLOCATED));

    if (retval != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "rx disconnect failed (%d)\n",
		    retval, 0, 0, 0, 0, 0);
	return retval;
    }

    return IX_SUCCESS;
}


IX_STATUS
ixEthAal5AppUtilsAtmVcRegisterConnect (IxAtmLogicalPort port,
				 unsigned vpi,
				 unsigned vci,
				 IxAtmdAccAalType aalType,
				 IxAtmServiceCategory atmService,
				 IxAtmRxQueueId rxQueueId,
				 IxAtmdAccRxVcRxCallback rxCallback,
				 unsigned int minimumReplenishCount,
				 IxAtmdAccTxVcBufferReturnCallback bufferFreeCallback,
				 IxAtmdAccRxVcFreeLowCallback rxFreeLowCallback,
				 IxAtmdAccUserId userId,
				 IxAtmConnId *rxConnId,
				 IxAtmConnId *txConnId)
{
    IX_STATUS retval;
    IxAtmmVc txVc;
    IxAtmmVc rxVc;
    unsigned int rxFreeQueueSize;
    unsigned rxChannelIdx;
    unsigned txChannelIdx;

    if (rxFreeChannelGet(&rxChannelIdx) == IX_FAIL)
    {
        ixOsServLog(LOG_ERROR, "Failed to get a free Rx channell\n", 0, 0, 0, 0, 0, 0);
        return IX_FAIL;
    }

    if (txFreeChannelGet(&txChannelIdx) == IX_FAIL)
    {
        ixOsServLog(LOG_ERROR, "Failed to get a free Tx channell\n", 0, 0, 0, 0, 0, 0);
        return IX_FAIL;
    }

    rxPort[rxChannelIdx] = port;
    txPort[txChannelIdx] = port;

    memset(&txVc, 0, sizeof(txVc));
    memset(&rxVc, 0, sizeof(rxVc));

    /* Setup Tx Vc descriptor */
    txVc.vpi = vpi;
    txVc.vci = vci;
    txVc.direction = IX_ATMM_VC_DIRECTION_TX;
    txVc.trafficDesc.atmService = atmService;
    txVc.trafficDesc.pcr = UTOPIA_PORT_RATE;

    /* Setup tx VC */
    retval = ixAtmmVcRegister (port,
			       &txVc,
			       &txVcId[txChannelIdx]);

    if (retval != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Failed to register Tx VC \n", 0, 0, 0, 0, 0, 0);
	return retval;
    }

    /* Setup Rx Vc descriptor */
    rxVc.vpi = vpi;
    rxVc.vci = vci;
    rxVc.direction = IX_ATMM_VC_DIRECTION_RX;
    rxVc.trafficDesc.atmService = atmService;

    /* Setup rx VC */
    retval = ixAtmmVcRegister (port,
			       &rxVc,
			       &rxVcId[rxChannelIdx]);

    if (retval != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Failed to register Rx VC \n", 0, 0, 0, 0, 0, 0);
	return retval;
    }

    /* Connect Tx to the Vc */
    retval = ixAtmdAccTxVcConnect (port,
				   vpi,
				   vci,
				   aalType,
				   userId,
				   bufferFreeCallback,
				   txConnId);

    if (retval != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Failed to connect to Tx VC \n", 0, 0, 0, 0, 0, 0);
	return retval;
    }

    txChannelInfo[txChannelIdx].connId = *txConnId;
    txChannelInfo[txChannelIdx].port = port;

    /* Connect Rx to the VC */
    retval = ixAtmdAccRxVcConnect (port,
				   vpi,
				   vci,
				   aalType,
				   rxQueueId,
				   userId,
				   rxCallback,
				   minimumReplenishCount,
				   rxConnId,
				   &rxNpeVcIdTable[rxChannelIdx]);

    if (retval != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Failed to connect to Rx VC \n", 0, 0, 0, 0, 0, 0);
	return retval;
    }

    rxChannelInfo[rxChannelIdx].connId = *rxConnId;
    rxChannelInfo[rxChannelIdx].port = port;

    retval = ixAtmdAccRxVcFreeEntriesQuery(*rxConnId,
                                           &rxFreeQueueSize);
    if (retval != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Failed to get the number of entries in rx free queue\n", 0, 0, 0, 0, 0, 0);
	return retval;
    }

 
    /* Replenish Rx buffers if callback is not NULL*/
    if (rxFreeLowCallback != NULL)
    {
	rxFreeLowCallback(userId);
	
	retval = ixAtmdAccRxVcFreeLowCallbackRegister (*rxConnId,
						       rxFreeQueueSize / 4,
						       rxFreeLowCallback);
    }
	
    if (retval != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Failed to register RxFreeLowCalback \n", 0, 0, 0, 0, 0, 0);
	return retval;
    }

    /* Register replish more callback */

    retval = ixAtmdAccRxVcEnable (*rxConnId);

    if (retval != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Failed to enable Rx VC \n", 0, 0, 0, 0, 0, 0);
	return retval;
    }

    return IX_SUCCESS;
}

IX_STATUS
ixEthAal5AppUtilsAtmVcUnregisterDisconnect (IxAtmConnId rxConnId, IxAtmConnId txConnId)
{
    IX_STATUS retval = IX_SUCCESS;
    unsigned int rxChannelIdx;
    unsigned int txChannelIdx;

    if (rxChannelFind(rxConnId, &rxChannelIdx) == IX_SUCCESS)
    {
        retval = rxVcDisconnect(rxConnId);
        if (retval != IX_SUCCESS)
        {
            ixOsServLog(LOG_ERROR, "Failed to disconnect an rx channel\n",
			0, 0, 0, 0, 0, 0);
            return IX_FAIL;
        }

        retval = ixAtmmVcDeregister(rxPort[rxChannelIdx], rxVcId[rxChannelIdx]);
        if (retval != IX_SUCCESS)
        {
            ixOsServLog(LOG_ERROR, "Failed to unregister an rx channel\n",
			0, 0, 0, 0, 0, 0);
            return IX_FAIL;
        }
        rxChannelInfo[rxChannelIdx].inUse = FALSE;
    }

    if (txChannelFind(txConnId, &txChannelIdx) == IX_SUCCESS)
    {
        retval = txVcDisconnect(txConnId);
        if (retval != IX_SUCCESS)
        {
            ixOsServLog(LOG_ERROR, "Failed to disconnect a tx channel\n",
			0, 0, 0, 0, 0, 0);
            return IX_FAIL;
        }

        retval = ixAtmmVcDeregister(txPort[txChannelIdx], txVcId[txChannelIdx]);
        if (retval != IX_SUCCESS)
        {
            ixOsServLog(LOG_ERROR, "Failed to unregister a tx channel\n",
			0, 0, 0, 0, 0, 0);
            return IX_FAIL;
        }
        txChannelInfo[txChannelIdx].inUse = FALSE;
    }

    return IX_SUCCESS;
}

IX_STATUS
ixEthAal5AppUtilsAtmAllVcsDisconnect (void)
{
    IX_STATUS retval;
    int channelIdx;

    for (channelIdx=0; channelIdx<IX_ATM_MAX_NUM_VC; channelIdx++)
    {
        retval = ixEthAal5AppUtilsAtmVcUnregisterDisconnect (rxChannelInfo[channelIdx].connId, 
						       txChannelInfo[channelIdx].connId);
        if (retval != IX_SUCCESS)
        {
	    ixOsServLog(LOG_ERROR, "Failed to unregister a channel\n",
			0, 0, 0, 0, 0, 0);
	    return IX_FAIL;
        }
    }
    return IX_SUCCESS;
}

BOOL 
ixEthAal5AppUtilsUtopiaFpgaStimulusAsMasterSet(void)
{
    UINT32 regVal;

    fpgaStimulusEnable();
    
    regVal= readStimulusReg(BSR_CTRL1);
    
    if(regVal == 0)
    {
	return FALSE;	
    }

    /* write to the sel_utp_n bit in the stimulus make sure this low enabled */
    regVal= regVal & 0xEFFF;

    regVal= regVal | 0x0600;  
    regVal= regVal ^ 0x0400;  
    writeStimulusReg(BSR_CTRL1, regVal); 
    
    regVal= readStimulusReg(BSR_CTRL1);

    /* mask out the dont cares and check the master is enabled */
    regVal = regVal & 0x600;

    if(regVal != 0x200)
    {
	return FALSE;
    }	

    return TRUE;
}
