/*
	File:		QTSprites.c
	
	Description: Contains code to create a sprite world and its associated sprites

	Author:		QuickTime team

	Copyright: 	 Copyright 1999 Apple Computer, Inc. All rights reserved.
	
	Disclaimer:	IMPORTANT:  This Apple software is supplied to you by Apple Computer, Inc.
				("Apple") in consideration of your agreement to the following terms, and your
				use, installation, modification or redistribution of this Apple software
				constitutes acceptance of these terms.  If you do not agree with these terms,
				please do not use, install, modify or redistribute this Apple software.

				In consideration of your agreement to abide by the following terms, and subject
				to these terms, Apple grants you a personal, non-exclusive license, under Apples
				copyrights in this original Apple software (the "Apple Software"), to use,
				reproduce, modify and redistribute the Apple Software, with or without
				modifications, in source and/or binary forms; provided that if you redistribute
				the Apple Software in its entirety and without modifications, you must retain
				this notice and the following text and disclaimers in all such redistributions of
				the Apple Software.  Neither the name, trademarks, service marks or logos of
				Apple Computer, Inc. may be used to endorse or promote products derived from the
				Apple Software without specific prior written permission from Apple.  Except as
				expressly stated in this notice, no other rights or licenses, express or implied,
				are granted by Apple herein, including but not limited to any patent rights that
				may be infringed by your derivative works or by other works in which the Apple
				Software may be incorporated.

				The Apple Software is provided by Apple on an "AS IS" basis.  APPLE MAKES NO
				WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
				WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
				PURPOSE, REGARDING THE APPLE SOFTWARE OR ITS USE AND OPERATION ALONE OR IN
				COMBINATION WITH YOUR PRODUCTS.

				IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
				CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
				GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
				ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION AND/OR DISTRIBUTION
				OF THE APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF CONTRACT, TORT
				(INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF APPLE HAS BEEN
				ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
				
	Change History (most recent first):

	   <46>	 	08/30/00	srk		Carbonization

*/


/************************************************************
*                                                           *
*    INCLUDE FILES                                          *
*                                                           *
*************************************************************/

#ifndef _QTSPRITES_
#include "QTSprites.h"


static void QTSprites_CreateSprites (SpriteWorld theSpriteWorld);

/************************************************************
*                                                           *
*    CONSTANTS                                              *
*                                                           *
*************************************************************/

enum {

numSprites				= 4,
numSpaceShipImages		= 24,
backgroundPictID		= 158,
firstSpaceShipPictID	= (backgroundPictID + 1),
spaceShipWidth			= 106,
spaceShipHeight			= 80

};

/************************************************************
*                                                           *
*    GLOBALS                                                *
*                                                           *
*************************************************************/

GWorldPtr						gSpritePlane = NULL;
Sprite							gSprites[numSprites];
Handle							gCompressedPictures[numSpaceShipImages];
ImageDescriptionHandle			gImageDescriptions[numSpaceShipImages];
Rect							gBounceBox;
Rect							gDestRects[numSprites];
Point							gDeltas[numSprites];
short							gCurrentImages[numSprites];
Handle							gCompressedPictures[numSpaceShipImages];
RGBColor						gBackgroundColor;

/************************************************************
*                                                           *
*    QTSprites_CreateSpriteWorld()                          *
*                                                           *
*    Creates our sprite world and its associated sprites    *
*                                                           *
*************************************************************/

SpriteWorld QTSprites_CreateSpriteWorld (Rect *windowBounds, CGrafPtr windowPtr)
{
	OSErr err;
	Rect bounds;
	SpriteWorld theSpriteWorld = NULL;

	// calculate the size of the destination
	bounds = *windowBounds;
	MacOffsetRect (&bounds, -1 * bounds.left, -1 * bounds.top);
	gBounceBox = bounds;
	MacInsetRect (&gBounceBox, 16, 16);

	// create a sprite layer graphics world 
	NewGWorld (&gSpritePlane, 0, &bounds, nil, nil, 0);
	if (gSpritePlane == NULL)
	{
		NewGWorld (&gSpritePlane, 0, &bounds, nil, nil, 0);
	}
	
	if (gSpritePlane)
	{

#ifdef TARGET_OS_WIN32
		LockPixels(GetGWorldPixMap(gSpritePlane));
#else
		LockPixels (GetPortPixMap(gSpritePlane));
#endif
		
		gBackgroundColor.red = 0x1234;
		gBackgroundColor.green = 0x0001;
		gBackgroundColor.blue = 0x0044;
		

		// create a sprite world
		err = NewSpriteWorld (&theSpriteWorld,		/* on return, a new sprite world */
								windowPtr,			/* destination */
								gSpritePlane,		/* sprite layer graphics world */
								&gBackgroundColor,	/* background color */
								nil);				/* graphics world to be used as the background. */
		
		// create sprites
		QTSprites_CreateSprites(theSpriteWorld);
	}

	return theSpriteWorld;
}

/************************************************************
*                                                           *
*    QTSprites_CreateSprites()                              *
*                                                           *
*    Creates the sprites for use with our sprite world      *
*                                                           *
*************************************************************/

void QTSprites_CreateSprites (SpriteWorld theSpriteWorld)
{
	short				lIndex;
	Handle				hCompressedData = NULL;
	PicHandle			hpicImage;
	OSErr				nErr;
	RGBColor			rgbcKeyColor;
	
	MacSetRect(&gDestRects[0], 132, 132, 132 + spaceShipWidth, 
		132 + spaceShipHeight);
	MacSetRect(&gDestRects[1], 50, 50, 50 + spaceShipWidth, 
		50 + spaceShipHeight);
	MacSetRect(&gDestRects[2], 100, 100, 100 + spaceShipWidth, 
		100 + spaceShipHeight);
	MacSetRect(&gDestRects[3], 130, 130, 130 + spaceShipWidth, 
		130 + spaceShipHeight);

	gDeltas[0].h = -3;
	gDeltas[0].v = 0;
	gDeltas[1].h = -5;
	gDeltas[1].v = 3;
	gDeltas[2].h = 4;
	gDeltas[2].v = -6;
	gDeltas[3].h = 6;
	gDeltas[3].v = 4;
	
	gCurrentImages[0] = 0;
	gCurrentImages[1] = numSpaceShipImages / 4;
	gCurrentImages[2] = numSpaceShipImages / 2;
	gCurrentImages[3] = numSpaceShipImages * 4 / 3;
	
	rgbcKeyColor.red = 0;
	rgbcKeyColor.green = 0;
	rgbcKeyColor.blue = 0;
	
	// recompress PICT images to make them transparent
	for (lIndex = 0; lIndex < numSpaceShipImages; lIndex++) 
	{
		ImageDescriptionHandle idh=nil;
		Handle  imageData=nil;

		hpicImage = (PicHandle)GetPicture(lIndex +
											firstSpaceShipPictID);
		DetachResource((Handle)hpicImage);

		nErr = RecompressPictureWithTransparency(hpicImage,
												&rgbcKeyColor, 
												nil,
												&gImageDescriptions[lIndex],
												&gCompressedPictures[lIndex]);
		KillPicture(hpicImage);
	}

	// create the sprites for the sprite world
	for (lIndex = 0; lIndex < numSprites; lIndex++) {
		MatrixRecord	matrix;

		SetIdentityMatrix(&matrix);
		
		matrix.matrix[2][0] = ((long)gDestRects[lIndex].left << 16);
		matrix.matrix[2][1] = ((long)gDestRects[lIndex].top << 16);

		nErr = NewSprite(&(gSprites[lIndex]),			/* on return, the ID of the new sprite */
						theSpriteWorld,					/* the sprite world for this sprite */
						gImageDescriptions[lIndex],		/* image description of the sprites image. */
						*gCompressedPictures[lIndex], 	/* sprite image data */
						&matrix,						/* sprite matrix */
						true,							/* is sprite visible? */
						lIndex); 						/* sprite layer */

	}
}

/************************************************************
*                                                           *
*    QTSprites_DisposeSpriteWorld()                         *
*                                                           *
*    Disposes our sprite world and its associated sprites   *
*                                                           *
*************************************************************/

void QTSprites_DisposeSpriteWorld (SpriteWorld theSpriteWorld)
{
	short	nIndex;

	// dispose of each sprites image data
	for (nIndex = 0; nIndex < numSprites; nIndex++)
	{
		if (gSprites[nIndex])
			DisposeSprite(gSprites[nIndex]);

		if (gCompressedPictures[nIndex])
			DisposeHandle(gCompressedPictures[nIndex]);
			
		if (gImageDescriptions[nIndex])
			DisposeHandle((Handle)gImageDescriptions[nIndex]);
	}
	
	// dispose of the sprite plane world
	if (gSpritePlane)
		DisposeGWorld(gSpritePlane);

	// dispose of the sprite world and associated graphics world
	if (theSpriteWorld)
		DisposeSpriteWorld(theSpriteWorld);
}


/************************************************************
*                                                           *
*    QTSprites_MoveSprites()                                *
*                                                           *
*    Animate the sprites by adjusting their properties via  *
*	 the SetSpriteProperty function							*
*                                                           *
*************************************************************/

void QTSprites_MoveSprites (void)
{
	short				nIndex;
	MatrixRecord		matrix;
	
	SetIdentityMatrix(&matrix);

	// for each sprite
	for (nIndex = 0; nIndex < numSprites; nIndex++) {
	
		// modify the sprites matrix
		MacOffsetRect(&gDestRects[nIndex], gDeltas[nIndex].h,
					gDeltas[nIndex].v);
		
		if ((gDestRects[nIndex].right >= gBounceBox.right) ||
			(gDestRects[nIndex].left <= gBounceBox.left))
			gDeltas[nIndex].h = -gDeltas[nIndex].h;
		
		if ((gDestRects[nIndex].bottom >= gBounceBox.bottom) ||
			(gDestRects[nIndex].top <= gBounceBox.top))
			gDeltas[nIndex].v = -gDeltas[nIndex].v;
		
		matrix.matrix[2][0] = ((long)gDestRects[nIndex].left << 16);
		matrix.matrix[2][1] = ((long)gDestRects[nIndex].top << 16);
		
		SetSpriteProperty(gSprites[nIndex],			/* the sprite for this operation */
							kSpritePropertyMatrix,	/* the property to be set */
							&matrix);				/* the new value for the property */
		
		// change the sprites image
		gCurrentImages[nIndex]++;
		if (gCurrentImages[nIndex] >= (numSpaceShipImages *
										(nIndex+1)))
		{
			gCurrentImages[nIndex] = 0;
		}
		
		SetSpriteProperty(gSprites[nIndex],
						kSpritePropertyImageDataPtr,
						*gCompressedPictures[gCurrentImages[nIndex] / (nIndex+1)]);
	}
}

#endif
