//////////
//
//	File:		QTStreamMsg.c
//
//	Contains:	Sample code for intercepting and issuing messages to the streaming controller bar.
//
//	Written by:	Tim Monroe
//
//	Copyright:	 1999 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <1>	 	06/11/99	rtm		first file
//
//
//	The movie controller indicates the status of streaming operations by sending action messages to itself.
//	You can intercept those messages in your movie controller action filter procedure by intercepting messages
//	of type mcActionShowStatusString. The parameter data is a pointer to a structure of type QTStatusStringRecord,
//	which is defined like this:
//	
//	struct QTStatusStringRecord {
//		long 						stringTypeFlags;
//		char *						statusString;
//	};
//	
//	You can identify streaming-specific messages by inspecting the stringTypeFlags field; if the bit
//	kStatusStringIsStreamingStatus is set, then the message pertains to streaming operations. In addition, if the bit
//	kStatusHasCodeNumber is set, then the high-order 16 bits of the stringTypeFlags field contain a result code. 
//	
//	The statusString field contains the actual text of the status message, which (if you do nothing to it) will be
//	displayed in the controller bar status area. You can (if you like) suppress the display of the message or change
//	the message to some other message.
//	
//////////

#include "QTStreamMsg.h"

char				*gMessageArray[] = MESSAGE_ARRAY;		// array of substitute messages
char				gStreamingMessage[kMaxMessageSize];		


//////////
//
// QTStreamMsg_IssueMessage
// Display the specified message in the controller bar of the streaming movie controlled by theMC.
//
//////////

void QTStreamMsg_IssueMessage (MovieController theMC, char *theMessage)
{
	QTStatusStringRecord	myRecord;
	
	myRecord.stringTypeFlags = kStatusStringIsStreamingStatus;
	myRecord.statusString = theMessage;
	
	MCDoAction(theMC, mcActionShowStatusString, &myRecord);
}


//////////
//
// QTStreamMsg_ActionFilterProc 
// Intercept some messages for the streaming movie movie controller.
//
// For the purposes of this sample code, we will interpret theRefCon as a Boolean value that indicates
// whether to suppress the display of the messages.
//
//////////

PASCAL_RTN Boolean QTStreamMsg_ActionFilterProc (MovieController theMC, short theAction, void *theParams, long theRefCon)
{
#pragma unused(theMC)

	Boolean			isHandled = false;			// false => allow controller to process the action
	Boolean			isSuppressed = (Boolean)theRefCon;
	
	switch (theAction) {
		
		case mcActionShowStatusString: {
			QTStatusStringPtr		myStatusPtr = (QTStatusStringPtr)theParams;
		
			// make sure it's a streaming status message
			if (!(myStatusPtr->stringTypeFlags & kStatusStringIsStreamingStatus))
				break;
			
			// save the original message for our application's own use elsewhere
			if (strlen(myStatusPtr->statusString) > 0)
				strcpy(gStreamingMessage, myStatusPtr->statusString);
			
			// intercept "Connecting", "Negotiating", and "Buffering" and substitute other messages
			if (strcmp(myStatusPtr->statusString, "Connecting") == 0)
				myStatusPtr->statusString = gMessageArray[kConnectingMsgIndex];
		
			if (strcmp(myStatusPtr->statusString, "Negotiating") == 0)
				myStatusPtr->statusString = gMessageArray[kNegotiatingMsgIndex];
		
			if (strcmp(myStatusPtr->statusString, "Buffering") == 0)
				myStatusPtr->statusString = gMessageArray[kBufferingMsgIndex];
		
			if (isSuppressed)
				isHandled = true;
				
			break;
		}
		
		// handle other mcAction messages here
		
		default:
			break;
			
	}	// switch (theAction)
	
	return(isHandled);	
}



