/*
	File:		IPDrawer.java
	
	Description:	This demo program shows how to display any QuickTime drawing object using Java's 
                        ImageProducing model.

	Author:		Apple Computer, Inc.

	Copyright: 	 Copyright 2002 Apple Computer, Inc. All rights reserved.
	
	Disclaimer:	IMPORTANT:  This Apple software is supplied to you by Apple Computer, Inc.
				("Apple") in consideration of your agreement to the following terms, and your
				use, installation, modification or redistribution of this Apple software
				constitutes acceptance of these terms.  If you do not agree with these terms,
				please do not use, install, modify or redistribute this Apple software.

				In consideration of your agreement to abide by the following terms, and subject
				to these terms, Apple grants you a personal, non-exclusive license, under Apples
				copyrights in this original Apple software (the "Apple Software"), to use,
				reproduce, modify and redistribute the Apple Software, with or without
				modifications, in source and/or binary forms; provided that if you redistribute
				the Apple Software in its entirety and without modifications, you must retain
				this notice and the following text and disclaimers in all such redistributions of
				the Apple Software.  Neither the name, trademarks, service marks or logos of
				Apple Computer, Inc. may be used to endorse or promote products derived from the
				Apple Software without specific prior written permission from Apple.  Except as
				expressly stated in this notice, no other rights or licenses, express or implied,
				are granted by Apple herein, including but not limited to any patent rights that
				may be infringed by your derivative works or by other works in which the Apple
				Software may be incorporated.

				The Apple Software is provided by Apple on an "AS IS" basis.  APPLE MAKES NO
				WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
				WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
				PURPOSE, REGARDING THE APPLE SOFTWARE OR ITS USE AND OPERATION ALONE OR IN
				COMBINATION WITH YOUR PRODUCTS.

				IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
				CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
				GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
				ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION AND/OR DISTRIBUTION
				OF THE APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF CONTRACT, TORT
				(INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF APPLE HAS BEEN
				ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
				
	Change History (most recent first):
            11/22/2002	md	new SampleCode revisions

*/

import quicktime.app.display.*;
import quicktime.app.image.*;
import quicktime.qd.*;
import quicktime.QTException;
import quicktime.std.image.Matrix;
import java.awt.*;

/**
 * This class will present the image that is produced by the given QTImageProducer.
 * If the QTImageProducer is presenting content that has a time or multi-frame element
 * then repaints will be scheduled when the IPDrawer is redrawn.
 * <P>
 * A typical usage of this class is as a client of a QTCanvas that is used within the
 * context of a swing container.
 */
public class IPDrawer extends ImageDrawer {
	/** 
	 * Creates an IPDrawer with the given QTImageProducer. The default size of the IPDrawer
	 * is the size of the QTImageProducer. 
	 */
	public IPDrawer (QTImageProducer ip) {
		super (ip.getSize(), null);
		this.ip = ip;
	}
	
//_________________________ INSTANCE VARIABLES
	private QTImageProducer ip;
	
//_________________________ INSTANCE METHODS
	/**
	 * Returns the QTImageProducer that produces the image for this Drawer
	 */
	public QTImageProducer getProducer () { return ip; }

	/**
	 * This method is called by the specified object when the instance of 
	 * the class that implements this interface is added to the object
	 * that is the source of the interest.
	 * @param interest the object that is to be the source of interest for the
	 * the object that implements this interface.
	 */
	public void addedTo (Object interest) {
		super.addedTo (interest);	//this will set the canv variable
		if (canv != null) {
			im = canv.createImage (ip);
			canv.prepareImage (im, canv);
		}
	}
	
	/**
	 * This method is called by the specified object when the instance of 
	 * the class that implements this interface is removed from the object
	 * that is the source of the interest.
	 * @param interest the object that was the source of interest for the
	 * the object that implements this interface.
	 */
	public void removedFrom (Object interest) {
		super.removedFrom (interest);
		if (canv == null)
			im = null;
	}

	/**
	 * QTCanvas calls this method when the client should redraw itself.
	 * If the canvas is able to discern that only a part of the client's 
	 * drawing area needs to be redrawn - then this area shall be passed in
	 * using the invalidRgn. Otherwise this will be null in which case the
	 * client should redraw itself entirely.
	 * @param invalidRgn the invalidRgn that the client should redraw
	 */
	public void redraw (Region invalidRgn) throws QTException {
		super.redraw (invalidRgn);
		
// this redraw should be done in another thread to free up the display thread
// so that the app is more repsonsive for the user
// see the sample code
		if (ip.isRedrawing() && canv != null) { 
			ip.redraw(invalidRgn);
		}
	}
}
