/*
	File:		ControlPanel.java
	
	Description:	This demo program shows how to display any QuickTime content within a java.awt display space using
                        the QTCanvas. It also demonstrates the use of the different resize options of the QTCanvas (with
                        the QTCanvas' alignment set to centre it in the display space).

	Author:		Apple Computer, Inc.

	Copyright: 	 Copyright 2002 Apple Computer, Inc. All rights reserved.
	
	Disclaimer:	IMPORTANT:  This Apple software is supplied to you by Apple Computer, Inc.
				("Apple") in consideration of your agreement to the following terms, and your
				use, installation, modification or redistribution of this Apple software
				constitutes acceptance of these terms.  If you do not agree with these terms,
				please do not use, install, modify or redistribute this Apple software.

				In consideration of your agreement to abide by the following terms, and subject
				to these terms, Apple grants you a personal, non-exclusive license, under Apples
				copyrights in this original Apple software (the "Apple Software"), to use,
				reproduce, modify and redistribute the Apple Software, with or without
				modifications, in source and/or binary forms; provided that if you redistribute
				the Apple Software in its entirety and without modifications, you must retain
				this notice and the following text and disclaimers in all such redistributions of
				the Apple Software.  Neither the name, trademarks, service marks or logos of
				Apple Computer, Inc. may be used to endorse or promote products derived from the
				Apple Software without specific prior written permission from Apple.  Except as
				expressly stated in this notice, no other rights or licenses, express or implied,
				are granted by Apple herein, including but not limited to any patent rights that
				may be infringed by your derivative works or by other works in which the Apple
				Software may be incorporated.

				The Apple Software is provided by Apple on an "AS IS" basis.  APPLE MAKES NO
				WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
				WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
				PURPOSE, REGARDING THE APPLE SOFTWARE OR ITS USE AND OPERATION ALONE OR IN
				COMBINATION WITH YOUR PRODUCTS.

				IN NO EVENT SHALL APPLE BE LIABLE FOR ANY SPECIAL, INDIRECT, INCIDENTAL OR
				CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
				GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
				ARISING IN ANY WAY OUT OF THE USE, REPRODUCTION, MODIFICATION AND/OR DISTRIBUTION
				OF THE APPLE SOFTWARE, HOWEVER CAUSED AND WHETHER UNDER THEORY OF CONTRACT, TORT
				(INCLUDING NEGLIGENCE), STRICT LIABILITY OR OTHERWISE, EVEN IF APPLE HAS BEEN
				ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
				
	Change History (most recent first):
            11/22/2002	md	new SampleCode revisions

*/

package duke;

import java.awt.*;
import java.awt.event.*;
import java.io.*;

import quicktime.io.QTFile;
import quicktime.util.QTUtils;
import quicktime.*;
import quicktime.app.*;
import quicktime.std.StdQTException;
import quicktime.std.movies.MovieController;
import quicktime.app.display.QTCanvas;

/**
 * A simple panel of AWT objects to demonstrate how AWT can be used to control a
 * QuickTime movie.
 */
public class ControlPanel extends Panel implements Errors {
//_________________________ CLASS METHODS
	public ControlPanel(Color bgColor, MovieScreen ms) {
		this.ms = ms;
	
		setBackground(bgColor);
		setLayout(new FlowLayout(FlowLayout.CENTER, 10, 3));
		setFont(new Font("Geneva", Font.BOLD, 10));
		
		Action actionListener = new Action();
		Item itemListener = new Item();
	
		// start, stop, loop and get a new movie
		movieButtons.setBackground(getBackground());
		movieButtons.setLayout(new GridLayout(4, 1, 3, 3));
		movieButtons.add(startButton);
		movieButtons.add(stopButton);
		loopButton.setState(false);
		movieButtons.add(loopButton);
		movieButtons.add(getMovieButton);
		add(movieButtons);
		
		startButton.addActionListener( actionListener );
		stopButton.addActionListener( actionListener );
		loopButton.addItemListener( itemListener );
		getMovieButton.addActionListener( actionListener );
		
		// choose a resizing option
		resizeButtons.setBackground(getBackground());
		resizeButtons.setLayout(new GridLayout(4, 2));
		resizeButtons.add(initButton);
		resizeButtons.add(integralButton);
		resizeButtons.add(horizontalButton);
		resizeButtons.add(aspectButton);
		resizeButtons.add(verticalButton);
		resizeButtons.add(freeButton);
		resizeButtons.add(performButton);
		add(resizeButtons);
			
		initButton.addItemListener( itemListener );
		integralButton.addItemListener( itemListener );
		horizontalButton.addItemListener( itemListener );
		aspectButton.addItemListener( itemListener );
		verticalButton.addItemListener( itemListener );
		freeButton.addItemListener( itemListener );
		performButton.addItemListener( itemListener );

		// choose a visibility option
		visiblity.setBackground(getBackground());
		visiblity.setLayout(new GridLayout(3, 1));
		visiblity.add(visibleButton);
		visiblity.add(invisibleButton);
		add(visiblity);
		
		visibleButton.addItemListener( itemListener );
		invisibleButton.addItemListener( itemListener );
	}

	public Dimension getPreferredSize () { return new Dimension (100, 130); }
//_________________________ INSTANCE VARIABLES
	private MovieScreen ms;

	private Panel  movieButtons   = new Panel();
	private Button startButton    = new Button("Start");
	private Button stopButton     = new Button("Stop");
	private Button getMovieButton = new Button("Get QT Media");

	private Checkbox loopButton = new Checkbox("Looping");

	private Panel         resizeButtons = new Panel();
	private CheckboxGroup checkBoxGrp   = new CheckboxGroup();
	private Checkbox      initButton    = new Checkbox("Initial Size or Smaller", checkBoxGrp, true);
	private Checkbox      freeButton    = new Checkbox("Allow Any Size", checkBoxGrp, false);
	private Checkbox      aspectButton  = new Checkbox("Maintain Aspect", checkBoxGrp, false);
	private Checkbox      performButton = new Checkbox("Performance Sizing", checkBoxGrp, false);
	private Checkbox      horizontalButton = new Checkbox("Horizontal Sizing", checkBoxGrp, false);
	private Checkbox      verticalButton = new Checkbox("Vertical Sizing", checkBoxGrp, false);
	private Checkbox      integralButton = new Checkbox("Integral Sizing", checkBoxGrp, false);

	private Panel         visiblity		 = new Panel();
	private CheckboxGroup checkBoxGrp2           = new CheckboxGroup();
	private Checkbox      visibleButton  = new Checkbox("Visible", checkBoxGrp2, true);
	private Checkbox      invisibleButton= new Checkbox("Invisible", checkBoxGrp2, false);

//_________________________ INSTANCE METHODS
	public void doGetMovie () {
		try {
			ms.stopPlayer ();
			QTFile qtf = QTFile.standardGetFilePreview(QTFile.kStandardQTFileTypes);
// Below is code to use the Standard Java File Dialog, i.e no fancy Quicktime preview dialog.
//			FileDialog fd = new FileDialog ((Frame)getParent(), "Choose a QuickTime file to view:");
//			QTFileTypesFilter f = new QTFileTypesFilter();
//			f.set3DAccept(false);
//			fd.setFilenameFilter(f);
//			fd.show();
//			String fileDir = fd.getDirectory();
//			String fileName = fd.getFile();
//			if (fileDir != null && fileName != null) {
//				QTFile qtf = new QTFile (fileDir + fileName);
				ms.getNewMovie(qtf);
//			}
		} catch (QTException e) {
		} catch (Exception e) {
		} finally { QTUtils.reclaimMemory(); }
	}
	
	public void doMove (float deltaX, float deltaY) {
		QTCanvas canv = ms.getCanvas();
		if (canv == null) return;	
		float x = canv.getAlignmentX() + deltaX;
		float y = canv.getAlignmentY() + deltaY;
		if (x < 0) x = 1;
		else if (x > 1) //catch floating point errors
			if (x < 1.09) x = 1; else x = 0;
		if (y < 0) y = 1;
		else if (y > 1) //catch floating point errors
			if (y < 1.09) y = 1; else y = 0;
		ms.getCanvas().setAlignment(x, y);
	}
	
	/** Handle all the AWT checkbox item events here. */
	class Item implements ItemListener {
		public void itemStateChanged( ItemEvent event ) {
			if ( event.getSource() == visibleButton ) {
				ms.getCanvas().setVisible(true);
			}
			else if (event.getSource() == invisibleButton) {
				ms.stopPlayer();
				ms.getCanvas().setVisible(false);
			}
			else if (event.getSource() == loopButton) {
				if (ms.isLooping()) {
					loopButton.setState(false);
					ms.setLooping(false);
				}
				else {
					loopButton.setState(true);
					ms.setLooping(true);
				}
			}
			else if (event.getSource() == initButton) {
				ms.getCanvas().setResizeFlag(QTCanvas.kInitialSize);
			}
			else if (event.getSource() == horizontalButton) {
				ms.getCanvas().setResizeFlag(QTCanvas.kHorizontalResize);
			}
			else if (event.getSource() == verticalButton) {
				ms.getCanvas().setResizeFlag(QTCanvas.kVerticalResize);
			}
			else if (event.getSource() == integralButton) {
				ms.getCanvas().setResizeFlag(QTCanvas.kIntegralResize);
			}
			else if (event.getSource() == freeButton) {
				ms.getCanvas().setResizeFlag(QTCanvas.kFreeResize);
			}
		}
	}
	
	/** Handle all the AWT button events here. */
	class Action implements ActionListener {
		public void actionPerformed  (ActionEvent event) {
			if ( event.getSource() == getMovieButton ) {
				doGetMovie();
			}
			else if (event.getSource() == stopButton) {
				ms.stopPlayer();
			}
			else if (event.getSource() == startButton) {
				ms.startPlayer();
			}
			else if (event.getSource() == aspectButton) {
				ms.getCanvas().setResizeFlag(QTCanvas.kAspectResize);
			}
			else if (event.getSource() == performButton) {
				ms.getCanvas().setResizeFlag(QTCanvas.kPerformanceResize);
			}
		}
	}
}
