/*
	File:		Queue.c

	Contains:	QuickTime Streaming sample application.

	Written by:	Scott Kuechle

	Copyright:	 1999 by Apple Computer, Inc., all rights reserved.

	Change History (most recent first):

	   <1>	 	11/20/98		srk		first file
	    
 
 NOTES:

 
 TO DO:

*/


/************************************************************
*                                                           *
*    INCLUDE FILES                                          *
*                                                           *
*************************************************************/

#include "Queue.h"


/************************************************************
*                                                           *
*    GLOBALS                                                *
*                                                           *
*************************************************************/

static	QHdr		gNewPresentationQ;	/* holds any new presentations which we have detected */
static	QHdr		gFreeQ;				/* our free queue - holds all available (unused) queue entries */
static	MyQElemRec	gQElem[kQItems];	/* available queue entries */


/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_InitQHdr()                                              */
/*                                                                  */
/*                                                                  */
/*    Initialize the queue head and tail to null entries            */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static void Queue_InitQHdr(QHdrPtr qHdrPtr)
{
	qHdrPtr->qHead = NULL;
	qHdrPtr->qTail = NULL;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_InitQHdr_()                                             */
/*                                                                  */
/*                                                                  */
/*    Initialize the "Free" and "Used" queues                       */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

void Queue_InitQHdr_(void)
{
	Queue_InitQHdr(&gNewPresentationQ);
	Queue_InitQHdr(&gFreeQ);
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_InitQ()                                                 */
/*                                                                  */
/*                                                                  */
/*    Fills the queue specified in the qHdrPtr with the desired     */
/*    number of queue items                                         */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static	void Queue_InitQ(QHdrPtr qHdrPtr, MyQElemRec qElemPtr[], int qItems)
{
	int i;

		for (i=0; i<qItems; ++i)
		{
				/* init queue elements first */
			qElemPtr->q.qLink = 0;
			qElemPtr->q.qType = 0;
			qElemPtr->presentation = 0;

				/* add queue element to the queue */
			Enqueue((QElemPtr)&qElemPtr[i], qHdrPtr);
		}
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_InitQ_()                                                */
/*                                                                  */
/*                                                                  */
/*    Fills the "Free" queue with the desired number of queue       */
/*    items                                                         */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

void Queue_InitQ_(void)
{
	Queue_InitQ(&gFreeQ, &gQElem, kQItems);
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_GetQElem()                                              */
/*                                                                  */
/*                                                                  */
/*    Gets (removes) a queue element from the queue specified in    */
/*    the qHdrPtr parameter                                         */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static	MyQElemRecPtr Queue_GetQElem(QHdrPtr qHdrPtr)
{
	QElemPtr qElemPtr;

		qElemPtr = qHdrPtr->qHead;
		if (qElemPtr != 0L)
		{
			OSErr err;

				err = Dequeue((QElemPtr)qElemPtr, qHdrPtr);
				if (err == noErr)
				{
					return ((MyQElemRecPtr)qElemPtr);
				}
				else
				{
					return NULL;
				}
		}
		else
		{
				/* no free items in the queue */
			return NULL;
		}
}


/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_GetFreeQElem()                                          */
/*                                                                  */
/*                                                                  */
/*    Gets (removes) a queue element from the Free queue            */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

MyQElemRecPtr Queue_GetFreeQElem(void)
{
	return (Queue_GetQElem(&gFreeQ));
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_GetUsedQElem()                                          */
/*                                                                  */
/*                                                                  */
/*    Gets (removes) a queue element from the Used queue            */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

MyQElemRecPtr Queue_GetUsedQElem(void)
{
	return(Queue_GetQElem(&gNewPresentationQ));
}


/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_PutElemIntoUsedQueue                                    */
/*                                                                  */
/*                                                                  */
/*    Add a queue element to the Used queue                         */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

void Queue_PutElemIntoUsedQueue(MyQElemRecPtr qElemPtr)
{
	Enqueue((QElemPtr)qElemPtr, &gNewPresentationQ);
}


/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_PutElemIntoUsedQueue                                    */
/*                                                                  */
/*                                                                  */
/*    Add a queue element to the Used queue                         */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

void Queue_PutElemIntoFreeQueue(MyQElemRecPtr qElemPtr)
{
	Enqueue((QElemPtr)qElemPtr, &gFreeQ);
}



/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_SetQElemPresValue                                       */
/*                                                                  */
/*                                                                  */
/*    Save a presentation value in a private queue element          */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

void Queue_SetQElemPresValue(MyQElemRecPtr qElemPtr,
						   QTSPresentation presentation)
{
	qElemPtr->presentation = presentation;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Queue_GetQElemPresValue                                       */
/*                                                                  */
/*                                                                  */
/*    Get the presentation value from a private queue element       */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

QTSPresentation Queue_GetQElemPresValue(MyQElemRecPtr qElemPtr)
{
	return(qElemPtr->presentation);
}