/*
	File:		QTCode.c

	Contains:	QuickTime Streaming sample application.

	Written by:	Scott Kuechle

	Copyright:	 1999 by Apple Computer, Inc., all rights reserved.

	Change History (most recent first):

	   <1>	 	11/20/98		srk		first file
	    
 
 NOTES:

 
 TO DO:

*/


/************************************************************
*                                                           *
*    INCLUDE FILES                                          *
*                                                           *
*************************************************************/

#include "QTCode.h"
#include "QTStreamingComponents.h"
#include "QuickTimeComponents.h"
#include "Queue.h"


/************************************************************
*                                                           *
*    CONSTANTS                                              *
*                                                           *
*************************************************************/

#define	BailNil(n)		if (!n) goto bail;
#define	BailError(n)	if (n) goto bail;

#define	kFullVolume		255



/* ---------------------------------------------------------------- */
/*                                                                  */
/*    QTCode_DoCreatePortAssociation()                              */
/*                                                                  */
/*                                                                  */
/*    Create a mac graphics port and associate it with a given      */
/*    window so QTML can draw into it                               */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

GrafPtr QTCode_DoCreatePortAssociation(void *	theWnd,
										Ptr 	storage,
										long 	flags)
{
		/* Associate a GrafPort with this window */
	return(CreatePortAssociation(theWnd, storage, flags));
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    QTCode_DoDestroyPortAssociation()                             */
/*                                                                  */
/*                                                                  */
/*    De-register a mac graphics port and its associated            */
/*    window                                                        */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

void QTCode_DoDestroyPortAssociation(HWND hwnd)
{
		/* Destroy the port association */
	DestroyPortAssociation((CGrafPtr)GetHWNDPort(hwnd));
}



/* ---------------------------------------------------------------- */
/*                                                                  */
/*    QTCode_DoQTInit()                                             */
/*                                                                  */
/*                                                                  */
/*    Initialize Quicktime and QuickTime Streaming                  */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

OSErr QTCode_DoQTInit()
{
	OSErr err = noErr;

			/* Initialize QuickTime Media Layer */
		err = InitializeQTML(0L);
		BailError(err);

			/* Initialize QuickTime Streaming Toolbox */
		err = InitializeQTS();
		BailError(err);

			/* Initialize QuickTime */
		err = EnterMovies();
		BailError(err);

	bail:
		return (err);
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    QTCode_QTCleanUp()                                            */
/*                                                                  */
/*                                                                  */
/*    De-initialize Quicktime and QuickTime Streaming               */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

void QTCode_QTCleanUp()
{
		/* Clean up */
	ExitMovies();
		/* notify QTS */
	TerminateQTS();
		/* notify QTML */
	TerminateQTML();
}



/* ---------------------------------------------------------------- */
/*                                                                  */
/*    QTCode_NewPresFromFile()                                      */
/*                                                                  */
/*                                                                  */
/*    Create a new presentation from an SDP file                    */
/*                                                                  */
/* ---------------------------------------------------------------- */

OSErr QTCode_NewPresFromFile(char				*filePath,			/* sdp file path */
							 QTSPresentation	*aPresentation)		/* return presentation value here */
{
	QTSNewPresentationParams	*presParams = NULL;
	FSSpec						fileSpec;
	OSErr						err;

			// initialize presentation value
		*aPresentation = NULL;

			// allocate parameter block memory
		presParams = (QTSNewPresentationParams	*)malloc(sizeof (struct QTSNewPresentationParams));
		BailNil(presParams);

			// Fill in an FSSpec.
		NativePathNameToFSSpec(filePath, &fileSpec, 0L);

			// fill in a presentation parameter block
		presParams->dataType			= kQTSFileDataType;
		presParams->data				= &fileSpec;
		presParams->dataLength			= sizeof(FSSpec);
		presParams->flags				= kQTSAutoModeFlag +	// configure the conductor with sourcers automatically
											 kQTSReceiveMediaFlag; 	// configure the conductor to receive

		presParams->timeScale			= 0L;
		presParams->notificationProc	= (QTSNotificationUPP)QTCode_NotificationProc;
		presParams->notificationRefCon	= NULL;
		presParams->editList			= NULL;
		presParams->mediaParams			= NULL;

			// create a new presentation
		err = QTSNewPresentation(presParams, aPresentation);
			// release memory we allocated for the parameter block
		free((void *)presParams);

		return(err);

bail:
		return -1;
}


/* ---------------------------------------------------------------- */
/*                                                                  */
/*    QTCode_ConfigurePresentation()                                */
/*                                                                  */
/*                                                                  */
/*    Configure the attributes of a presentation such as volume,    */
/*    dimensions, etc.                                              */
/*                                                                  */
/* ---------------------------------------------------------------- */

OSErr QTCode_ConfigurePresentation(QTSPresentation	aPresentation,
								  QTSStream			theStream,
								  CGrafPtr			aWindow)
{
	OSErr	result;

			/* tell the presentation where to draw visual data */
		result = QTSPresSetGWorld(aPresentation,
									theStream,
									(CGrafPtr) aWindow,
									nil);
		BailError(result);

			/* tell the presentation how big to draw visual data */
		result = QTSPresSetDimensions(aPresentation,
									theStream,
									(kPresWidth << 16),
									(kPresHeight << 16) );
		BailError(result);

			/* tell the presentation how loud to play the audio data */
		result = QTSPresSetVolumes(aPresentation,
									theStream,
									kFullVolume,
									kFullVolume);
		BailError(result);

			/* start the presentation with a preroll */
		result = QTSPresPreroll(aPresentation,
								 theStream,
								 1,					/* inTimeValue */
								 (Fixed)(1<<16),	/* inRate */
								 kQTSTimeBaseRunningControlFlag);
	
	bail:
		;
		
		return result;
}


/* ---------------------------------------------------------------- */
/*                                                                  */
/*    QTCode_DestroyPresentation()                                  */
/*                                                                  */
/*                                                                  */
/*    Stop and dispose a presentation                               */
/*                                                                  */
/* ---------------------------------------------------------------- */

OSErr QTCode_DestroyPresentation(QTSPresentation	aPresentation,
								QTSStream			theStream)
{
	OSErr	result;

			// stop the presentation
		result = QTSPresStop(aPresentation, theStream, 0L);
		BailError(result);
		
			// dispose the presentation
		result = QTSDisposePresentation(aPresentation, 0);
	
	bail:
		;

		return result;
}





/* ---------------------------------------------------------------- */
/*                                                                  */
/*    QTCode_NotificationProc()                                     */
/*                                                                  */
/*                                                                  */
/*    Notification procedure for our presentations                  */
/*                                                                  */
/* ---------------------------------------------------------------- */
ComponentResult QTCode_NotificationProc(ComponentResult inErr,
										OSType			inNotificationType,
										void			*inNotificationParams,
										UInt32			inRefCon)
{
	
	switch (inNotificationType)
	{
		case kQTSErrorNotification:
			break;

		case kQTSNewPresDetectedNotification:
			break;

		case kQTSNewPresentationNotification:
			{
				MyQElemRecPtr qElemPtr = NULL;

						/* get a private queue element to store the 
							presentation information for later processing
							in our CheckforNewPresentations() function */
					qElemPtr = Queue_GetFreeQElem();
					if (qElemPtr != NULL)
					{
							/* store the presentation value in our private queue
								element where we can retrieve it later */
						Queue_SetQElemPresValue(qElemPtr, 
												(QTSPresentation)inNotificationParams);

							/* put the presentation Q element into the "new" 
								queue for later processing in our main event loop.
								See the CheckForNewPresentations() function
								where we will get the presentation value and
								create a new window for it */
						Queue_PutElemIntoUsedQueue(qElemPtr);
					}
			}
			break;

		case kQTSNewStreamNotification:
			{
				QTSNewStreamParams *streamParms =
					(QTSNewStreamParams *)inNotificationParams;

				/* do any custom processing here */
			}
			break;

		case kQTSStreamChangedNotification:
			break;

		case kQTSStreamGoneNotification:
			break;

		case kQTSPrerollAckNotification:
			{
				QTSPresentation	thePresentation;
				QTSStream		*theStream;

						/* get the stream associated with this
							notification */
					theStream = (QTSStream *)inNotificationParams;
						/* get the presentation associated with 
							this stream */
					thePresentation = QTSGetStreamPresentation(*theStream);
					if (thePresentation != NULL)
					{
							/* start the presentation */
						Streaming_HandlePrerollAck(thePresentation,
													kQTSAllStreams);
					}
			}
			break;

		case kQTSStartAckNotification:
			break;

		case kQTSStopAckNotification:
			break;

		case kQTSStatusNotification:
			break;

		default:
			break;
	}


	return noErr;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    Streaming_HandlePrerollAck                                    */
/*                                                                  */
/*                                                                  */
/*    Starts the presentation when the preroll ack is received      */
/*                                                                  */
/* ---------------------------------------------------------------- */
void Streaming_HandlePrerollAck(QTSPresentation aPresentation,
								QTSStream		theStream)
{
		// After the PrerollAck comes in, THEN start broadcasting

	QTSPresStart( aPresentation, theStream, 0L );

	return;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    QTCode_UpdatePresentation()                                   */
/*                                                                  */
/*                                                                  */
/*    Force an update of our presentation window                    */
/*                                                                  */
/* ---------------------------------------------------------------- */

void QTCode_UpdatePresentation(QTSPresentation	aPresentation)
{
	RgnHandle	clipRegion = NULL;

	
		QTSPresIdle(aPresentation, nil);

		#if 0
				clipRegion = NewRgn();
				if (clipRegion != NULL)
				{
					GetClip(clipRegion);
					QTSPresInvalidateRegion	(aPresentation,
											clipRegion);
					DisposeRgn(clipRegion);

				}
		#endif
}
