//////////
//
//	File:		QTFullScreen.c
//
//	Contains:	Functions to display full-screen QuickTime movies.
//
//	Written by:	Tim Monroe
//
//	Copyright:	 1998 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <2>	 	03/17/98	rtm		finally got back to this; now it's working on Mac and Windows
//	   <1>	 	12/22/97	rtm		first file
//
//	This file contains functions that illustrate how to play QuickTime movies full screen. The
//	key elements to displaying full screen movies are the calls BeginFullScreen and EndFullScreen,
//	introduced in QuickTime 2.5. Here we open a QuickTime movie, configure it to play full screen,
//	associate a movie controller, and then let the controller handle events. Your application should
//	call the function QTFullScreen_EventLoopAction in its event loop (on MacOS) or when it gets idle
//	events (on Windows).
// 
//////////

#include "QTFullScreen.h"

// gloabl variables
WindowPtr					gFullScreenWindow = NULL;		// the full-screen window
MovieController				gMC = NULL;						// movie controller for the full-screen window
Ptr							gRestoreState = NULL;			// restore state; used when closing the full-screen window


//////////
//
// QTFullScreen_PlayOnFullScreen
//
//////////

OSErr QTFullScreen_PlayOnFullScreen (void)
{
	FSSpec				myFSSpec;
	Movie				myMovie = NULL;
	short				myRefNum = 0;
	SFTypeList			myTypeList = {MovieFileType, 0, 0, 0};
	StandardFileReply	myReply;
	long				myFlags = fullScreenDontChangeMenuBar | fullScreenAllowEvents;
	OSErr				myErr = noErr;
	
	StandardGetFilePreview(NULL, 1, myTypeList, &myReply);
	if (!myReply.sfGood)
		goto bail;
	
	// make an FSSpec record
	FSMakeFSSpec(myReply.sfFile.vRefNum, myReply.sfFile.parID, myReply.sfFile.name, &myFSSpec);

	myErr = OpenMovieFile(&myFSSpec, &myRefNum, fsRdPerm);
	if (myErr != noErr)
		goto bail;

	// now fetch the first movie from the file
	myErr = NewMovieFromFile(&myMovie, myRefNum, NULL, NULL, newMovieActive, NULL);
	if (myErr != noErr)
		goto bail;
	
	CloseMovieFile(myRefNum);

	// set up for full-screen display
	myErr = BeginFullScreen(&gRestoreState, NULL, 0, 0, &gFullScreenWindow, NULL, myFlags); 

#if TARGET_OS_WIN32
	// on Windows, set a window procedure for the new window and associate a port with that window
	QTMLSetWindowWndProc(gFullScreenWindow, QTFullScreen_HandleMessages);
	CreatePortAssociation(GetPortNativeWindow(gFullScreenWindow), NULL, 0L);
#endif
	
	SetMovieGWorld(myMovie, (CGrafPtr)gFullScreenWindow, GetGWorldDevice((CGrafPtr)gFullScreenWindow));
	SetMovieBox(myMovie, &gFullScreenWindow->portRect);

	// create the movie controller
	gMC = NewMovieController(myMovie, &gFullScreenWindow->portRect, 0);

bail:
	return(myErr);
}


//////////
//
// QTFullScreen_RestoreScreen
//
//////////

OSErr QTFullScreen_RestoreScreen (void)
{
	OSErr		myErr = noErr;
	
#if TARGET_OS_WIN32	
	DestroyPortAssociation((CGrafPtr)gFullScreenWindow);
#endif

	DisposeMovieController(gMC);
	myErr = EndFullScreen(gRestoreState, 0L); 
	
	return(myErr);
}


//////////
//
// QTFullScreen_EventLoopAction
// Do any required event loop action processing.
//
//////////

OSErr QTFullScreen_EventLoopAction (EventRecord *theEvent)
{
	return(MCIsPlayerEvent(gMC, theEvent));
}


#if TARGET_OS_WIN32
//////////
//
// QTFullScreen_HandleMessages
// Handle Windows messages for the full-screen window.
// 
//////////

LRESULT CALLBACK QTFullScreen_HandleMessages (HWND theWnd, UINT theMessage, UINT wParam, LONG lParam)
{
	MSG				myMsg = {0};
	EventRecord		myMacEvent;
	LONG			myPoints = GetMessagePos();

	myMsg.hwnd = theWnd;
	myMsg.message = theMessage;
	myMsg.wParam = wParam;
	myMsg.lParam = lParam;
	myMsg.time = GetMessageTime();
	myMsg.pt.x = LOWORD(myPoints);
	myMsg.pt.y = HIWORD(myPoints);

	// translate the Windows message to a Mac event
	WinEventToMacEvent(&myMsg, &myMacEvent);

	// pass the Mac event to the movie controller
	QTFullScreen_EventLoopAction(&myMacEvent);
		
	return(DefWindowProc(theWnd, theMessage, wParam, lParam));
}
#endif	// TARGET_OS_WIN32



