//////////
//
//	File:		QTFlattenToHandle.c
//
//	Contains:	Handle data handler sample code.
//
//	Written by:	Tim Monroe
//				Based on the existing code revised by Deeje Cooley.
//
//	Copyright:	 1998 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <1>	 	04/09/98	rtm		first file; integrated existing code with shell framework
//	 
//
// A data handler is a component (of type DataHandlerType) that is responsible for reading and writing
// a media's data. In other words, a data handler provides data input and output services to the media's
// media handler. Originally, QuickTime included a file data handler. QuickTime version 2.0 introduced
// the handle data handler (component subtype HandleDataHandlerSubType), which allows you to play movie
// data stored in memory rather than in a file. This sample code shows how to work with the handle data
// handler.
//
// Here, we will open a movie file and then flatten the movie data into a handle. Then we will play the
// movie from the handle. The essential step is to create a data reference record describing the handle
// and then pass that record, instead of an FSSpec record, to FlattenMovieData. To do this, set the
// flattenFSSpecPtrIsDataRefRecordPtr flag when calling FlattenMovieData.
//
//////////


#include "QTFlattenToHandle.h"

//////////
//
// QTHandle_PromptUserForMovieFileAndFlattenToHandle
// Have the user select a movie file, open the movie file, and write the movie data
// into a handle. Then play the memory-based movie.
//
//////////

OSErr QTHandle_PromptUserForMovieFileAndFlattenToHandle (void)
{
	SFTypeList				myTypeList;
	StandardFileReply		myReply;
	short					myResRefNum;
	Movie					mySrcMovie;
	Movie					myMemMovie;		// the movie in our handle
	Handle					myHandle;		// the handle that contains the movie data
	long					mySize = 0L;
	long					myFlags = 0L;
	DataReferenceRecord		myDataRefRec;
	FSSpec					*myFile;
	OSErr					myErr = noErr;
	
	// have the user select a movie file;
	myTypeList[0] = MovieFileType;

	StandardGetFilePreview(NULL, 1, myTypeList, &myReply);
	if (!myReply.sfGood)
		goto bail;

	// open the movie file and create a movie from that file
	myErr = OpenMovieFile(&myReply.sfFile, &myResRefNum, fsRdPerm);
	if (myErr != noErr)
		goto bail;
	
	myErr = NewMovieFromFile(&mySrcMovie, myResRefNum, NULL, NULL, newMovieActive, NULL);
	if (myErr != noErr)
		goto bail;
		
	// get the size of the movie data
	mySize = GetMovieDataSize(mySrcMovie, 0, GetMovieDuration(mySrcMovie));
	
	// create a handle of the appropriate size
	myHandle = NewHandleClear(mySize);
	if (myHandle == NULL)
		goto bail;
		
	HLockHi(myHandle);

	// fill in the data reference record
    myDataRefRec.dataRefType = HandleDataHandlerSubType;
    myDataRefRec.dataRef = myHandle;

	myFlags = flattenFSSpecPtrIsDataRefRecordPtr;
    myFile = (FSSpec*)&myDataRefRec;

	// flatten the source movie into the handle
    myMemMovie = FlattenMovieData(mySrcMovie, myFlags, myFile, 0L, smSystemScript, 0L);

	HUnlock(myHandle);

	// play the movie once thru
	myErr = QTHandle_PlayMovieResource(myMemMovie);

bail:
	if (myResRefNum != 0)
		CloseMovieFile(myResRefNum);

	if (myHandle != NULL)
		DisposeHandle(myHandle);

	if (mySrcMovie != NULL)
		DisposeMovie(mySrcMovie);
	
	return(myErr);
}


//////////
//
// QTHandle_PlayMovieResource
// Play a movie.
//
//////////

OSErr QTHandle_PlayMovieResource (Movie theMovie)
{
	WindowPtr				myWindow = NULL;
	Rect					myBounds = {50, 50, 100, 100};
	Rect					myRect;
	OSErr					myErr = memFullErr;

	myWindow = NewCWindow(NULL, &myBounds, "\pMovie Window", false, 0, (WindowPtr)-1, false, 0);
	if (myWindow == NULL)
		goto bail;
		
	myErr = noErr;
	
	MacSetPort(myWindow);

	GetMovieBox(theMovie, &myRect);
	MacOffsetRect(&myRect, -myRect.left, -myRect.top);
	SetMovieBox(theMovie, &myRect);

	SizeWindow(myWindow, myRect.right, myRect.bottom, false);
	MacShowWindow(myWindow);

	SetMovieGWorld(theMovie, (CGrafPtr)myWindow, NULL);
	GoToBeginningOfMovie(theMovie);
	MoviesTask(theMovie, 0);
	StartMovie(theMovie);
	
	myErr = GetMoviesError();
	if (myErr != noErr)
		goto bail;
	
	while (!IsMovieDone(theMovie))
		MoviesTask(theMovie, 0);

bail:
	if (theMovie != NULL)
		DisposeMovie(theMovie);

	if (myWindow != NULL)
		DisposeWindow(myWindow);
		
	return(myErr);
}
