/*
	File:		EffectUtilities.c

	Contains:	Tween utilities to help you write effect components

	Written by:	Tom Dowdy

	Copyright:	 1997-1998 by Apple Computer, Inc., all rights reserved.
	
*/

#include <ImageCodec.h>
#include <NumberFormatting.h>
#include "EffectUtilities.h"
#include "BltMacros.h"

// ----------------------------------------------------------------------------------------
int LocalMemCmp(Ptr a, Ptr b, Size length)
{
	while (length--)
		{
		if (*a++ != *b++)
			return(1);
		}
		
	return(0);

} // LocalMemCmp

// ----------------------------------------------------------------------------------------
OSErr CreateTweenRecord(
			TweenGlobals *glob, 			// input: our globals
			TweenContainerRecord *pTween, 	// input/output: tween record to initialize
			OSType paramterType,			// input: where to find parameter (type)
			long parameterID,				// input: where to find parameter (ID)
			Size parameterSize, 			// input: size of parameters
			OSType dataType,				// input: type of tween, 0 for none
			void *startValue,				// input: pointer or start value of default (for long size, just the data)
			void *endValue,					// input: pointer or end value of default (for long size, just the data)
			TimeValue duration)				// input: duration of the tween
{
	OSErr	anErr = noErr;

	// all done?
	if (pTween->readyToGo)
		return(noErr);
		
	/* If we don't have the data handle, make one */
	if (pTween->tweenData == nil)
		{
		pTween->tweenData = NewHandleClear(parameterSize);
		anErr = MemError();
		if (anErr == noErr)
			{
			if (parameterSize == sizeof(unsigned long))
				((unsigned long*)(*(pTween->tweenData)))[0] = (unsigned long) EndianU32_NtoB(startValue);
			else
				BlockMoveData((Ptr) startValue, *(pTween->tweenData), parameterSize);
			}
		}
		
	/* If we don't have the tweeners set up yet, do so here */
	if ( (anErr == noErr) && (pTween->tween == nil) )
		{
		QTAtom	atom;
		
		// find the parameter atom, if it exists
		atom = QTFindChildByID ( glob->parameters, kParentAtomIsContainer,
								paramterType, parameterID, nil);
		if ( (atom) && (QTCountChildrenOfType(glob->parameters, atom, kTweenData) > 0) )
			{
			anErr = QTNewTween(&pTween->tween, glob->parameters, atom, duration);
			glob->atLeastOneTweener = true;
			}
		else
			{
			// okay, we either have a constant value, or need to default
			if (atom)
				{
				anErr = QTCopyAtomDataToHandle(glob->parameters, atom, pTween->tweenData);
				#if TARGET_RT_LITTLE_ENDIAN
				switch (dataType)
				{
					case kParameterTypeDataLong:
					case kParameterTypeDataFixed:
						**(long**)(pTween->tweenData) = EndianS32_BtoN(**(long**)(pTween->tweenData));
						break;
					case kParameterTypeDataRGBValue:
						MyEndianRGBColor(**(RGBColor**)pTween->tweenData);
						break;
					case kParameterTypeDataDouble:
						MyEndianQTFloatDouble(**pTween->tweenData);
						break;
				}
				#endif
				}
			else
				{
				if (dataType != 0)
					{
					QTAtom	tweenAtom;
					Boolean	valuesAreSame = false;
					
					if (glob->defaultParameters == nil)
						anErr = QTNewAtomContainer(&glob->defaultParameters);
					
					anErr = QTInsertChild(glob->defaultParameters, kParentAtomIsContainer, kTweenEntry, 0, 0,
													0, nil, &tweenAtom);

					if (anErr == noErr) 
					{
						dataType = EndianU32_NtoB(dataType);
						QTInsertChild(glob->defaultParameters, tweenAtom, kTweenType, 0, 0,
													sizeof(dataType), &dataType, nil);
					}
					
					if (anErr == noErr)
						{
						SetHandleSize(pTween->tweenData, parameterSize * 2);
						anErr = MemError();
						}

					if (anErr == noErr)
						{
						HLock(pTween->tweenData);
						if (parameterSize == sizeof(unsigned long))
							{
							((unsigned long*)(*(pTween->tweenData)))[0] = (unsigned long) EndianU32_NtoB(startValue);
							((unsigned long*)(*(pTween->tweenData)))[1] = (unsigned long) EndianU32_NtoB(endValue);
							
							if (startValue == endValue)
								valuesAreSame = true;
							}
						else
							{
							BlockMoveData((Ptr) startValue, *(pTween->tweenData), parameterSize);
							BlockMoveData((Ptr) endValue, (*(pTween->tweenData))+parameterSize, parameterSize);
							
							if (LocalMemCmp(startValue, endValue, parameterSize) == 0)
								valuesAreSame = true;
							}
						anErr = QTInsertChild(glob->defaultParameters, tweenAtom, kTweenData, 1, 0,
												parameterSize*2, *(pTween->tweenData), nil);
						HUnlock(pTween->tweenData);
						SetHandleSize(pTween->tweenData, parameterSize);
						}
					if ( anErr == noErr )
						{
						if (valuesAreSame)
							{
							// flip back to native format, so effect can just read from it
							((unsigned long*)(*(pTween->tweenData)))[0] = (unsigned long) startValue;
							}
						else
							{
							anErr = QTNewTween(&pTween->tween, glob->defaultParameters, tweenAtom, duration);
							glob->atLeastOneTweener = true;
							}
						}
					} // we need to create a default tween
				else
					{
					// note that we store in native format, so that effect can just read from it
					if (parameterSize == sizeof(unsigned long))
						((unsigned long*)(*(pTween->tweenData)))[0] = (unsigned long) startValue;
					else
						BlockMoveData((Ptr) startValue, *(pTween->tweenData), parameterSize);
					} // we don't have a tween, just a default single value
					
				} // if we default the value
			
			} // if we don't have existing tween
			
		} // need to make the tween
			
	if (anErr == noErr)
		pTween->readyToGo = true;
		
	return(anErr);
	
} // CreateTweenRecord

// ----------------------------------------------------------------------------------------
void DisposeTweenRecord(TweenContainerRecord *pTween)
{
	if (pTween->tween)
		{
		QTDisposeTween(pTween->tween);
		pTween->tween = nil;
		}
	if (pTween->tweenData)
		{
		DisposeHandle(pTween->tweenData);
		pTween->tweenData = nil;
		}

	pTween->readyToGo = false;
	
} // DisposeTweenRecord

// ----------------------------------------------------------------------------------------
OSErr InitializeTweenGlobals(TweenGlobals *pTweenGlobals, CodecDecompressParams *p)
{
	return PtrToHand(p->data, (Handle*)&(pTweenGlobals->parameters), p->bufferSize);
	
} // InitializeTweenGlobals

// ----------------------------------------------------------------------------------------
void DisposeTweenGlobals(TweenGlobals *pTweenGlobals)
{
	if (pTweenGlobals->defaultParameters)
		{
		QTDisposeAtomContainer(pTweenGlobals->defaultParameters);
		pTweenGlobals->defaultParameters = nil;
		}
	pTweenGlobals->atLeastOneTweener = false;
	if (pTweenGlobals->parameters)
		{
		DisposeHandle((Handle)pTweenGlobals->parameters);
		pTweenGlobals->parameters = nil;
		}
	
} // DisposeTweenGlobals

