//////////
//
//	File:		VRMovies.c
//
//	Contains:	Support for QuickTime movie playback in VR nodes.
//
//	Written by:	Tim Monroe
//				Some code borrowed from QTVRSamplePlayer by Bryce Wolfson.
//
//	Copyright:	 1996-1998 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <8>	 	05/04/98	rtm		added automatic rotation of movie
//	   <7>	 	03/06/97	rtm		started to implement video masking
//	   <6>	 	03/05/97	rtm		added VRMoov_SetChromaColor; added fChromaColor to app data record
//	   <5>	 	03/04/97	rtm		fixed compositing problems at back buffer edges
//	   <4>	 	03/03/97	rtm		added VRMoov_SetVideoGraphicsMode to handle compositing
//									without using an offscreen GWorld
//	   <3>	 	02/27/97	rtm		further development: borrowed some ideas from QTVRSamplePlayer;
//									added VRMoov_SetEmbeddedMovieWidth etc.
//	   <2>	 	12/12/96	rtm		further development: borrowed some ideas from BoxMoov demo 
//	   <1>	 	12/11/96	rtm		first file 
//	   
// This code draws the QuickTime movie frames into the back buffer, either directly
// or indirectly (via an offscreen GWorld). Direct drawing gives the best performance,
// but indirect drawing is necessary for some visual effects. 
//
//////////

// TO DO:
// + finish video masking by custom 'hide' hot spots (so video goes *behind* such hot spots)
// + fix: when compositing (either w or w/o offscreen buffer) and movie is not initially visible
//   and movie rect intersects bb rect edge:
//	 panning to movie shows some junk in movie rect when movie first visible.
// + fix: when compositing w/o offscreen buffer, there's a blanking at repeat of movie


// header files
#include <Movies.h>
#include <MediaHandlers.h>
#include <QDOffscreen.h>
#include <ColorPicker.h>
#include <FixMath.h>
#include "QTVRUtilities.h"
#include "VRMovies.h"

// constants
const RGBColor			kClearColor = {0x0000, 0xffff, 0x0000};		// the default chroma key color
const RGBColor			kBlackColor = {0x0000, 0x0000, 0x0000};
const RGBColor			kWhiteColor = {0xffff, 0xffff, 0xffff};

// global variables
UserEventUPP			gColorFilterUPP = NULL;						// UPP to our custom color picker dialog event filter


//////////
//
// VRMoov_InitWindowData
// Initialize any window-specific data.
//
//////////

ApplicationDataHdl VRMoov_InitWindowData (WindowObject theWindowObject)
{
	ApplicationDataHdl	myAppData;
	
	myAppData = (ApplicationDataHdl)NewHandleClear(sizeof(ApplicationDataRecord));
	if (myAppData != NULL) {
		(**myAppData).fMovie = NULL;
		(**myAppData).fMovieGWorld = NULL;
		(**myAppData).fMoviePixMap = NULL;
		(**myAppData).fMovieCenter.x = 0.0;
		(**myAppData).fMovieCenter.y = 0.0;
		(**myAppData).fMovieScale = 1.0;
		(**myAppData).fMovieWidth = kDefaultEmbMovieWidth;
		(**myAppData).fUseMovieGWorld = false;
		(**myAppData).fUseMovieCenter = true;
		(**myAppData).fQTMovieHasSound = false;
		(**myAppData).fCompositeMovie = false;
		(**myAppData).fUseHideRegion = true;
		(**myAppData).fBackBufferProc = NULL;
		(**myAppData).fChromaColor = kBlackColor;//kClearColor;
		(**myAppData).fHideRegion = NULL;
	}
	
	if (gColorFilterUPP == NULL)
		gColorFilterUPP = NewUserEventProc(VRMoov_ColorDialogEventFilter);

	return(myAppData);
}


///////////
//
// VRMoov_GetEmbeddedMovie
// Get the QuickTime movie to be embedded in a panorama.
// Returns a Boolean to indicate success (true) or failure (false).
//
//////////

Boolean VRMoov_GetEmbeddedMovie (WindowObject theWindowObject)
{
	StandardFileReply		myReply;
	short					myNumTypes = 1;
	SFTypeList				myTypes = {MovieFileType, 0};

	// do some preliminary parameter checking
	if (theWindowObject == NULL)
		return(false);
	
	if (!IsWindowObjectOurs(theWindowObject))
		return(false);
	
	// elicit the movie file from user
	StandardGetFilePreview(NULL, myNumTypes, myTypes, &myReply);
	if (!myReply.sfGood) {
		VRMoov_DumpEmbeddedMovie(theWindowObject); 	// clean up any existing embedded movie
		return(false);
	}
		
	return(VRMoov_LoadEmbeddedMovie(&myReply.sfFile, theWindowObject));
}


//////////
//
// VRMoov_LoadEmbeddedMovie
// Load the QuickTime movie in the specified file.
// Returns a Boolean to indicate success (true) or failure (false).
//
//////////

Boolean VRMoov_LoadEmbeddedMovie (FSSpec *theMovieFile, WindowObject theWindowObject)
{
	OSErr					myErr = noErr;
	short					myMovieFileRef;
	Movie					myMovie;
	QTVRInstance			myInstance = NULL;
	GWorldPtr				myGWorld = NULL;
	CGrafPtr				mySavedPort;
	GDHandle				mySavedGDevice;
	Rect					myRect;
	ApplicationDataHdl		myAppData;
	
	// open the movie
	myErr = OpenMovieFile(theMovieFile, &myMovieFileRef, fsRdPerm);
	if (myErr != noErr)
		return(false);

	myErr = NewMovieFromFile(&myMovie, myMovieFileRef, NULL, (StringPtr)NULL, newMovieActive, NULL);
	if ((myErr != noErr) || (myMovie == NULL))
		return(false);

	if (myMovieFileRef != 0)
		CloseMovieFile(myMovieFileRef);
		
	GetMovieBox(myMovie, &myRect);
	MacOffsetRect(&myRect, -myRect.left, -myRect.top);
	SetMovieBox(myMovie, &myRect);
	
	// keep track of the movie and movie rectangle (in our app-specific data structure)
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData != NULL) {
		(**myAppData).fMovie = myMovie;
		(**myAppData).fMovieRect = myRect;
		
		// set the movie matrix to rotate the movie image
		SetIdentityMatrix(&(**myAppData).fRotationMatrix);
	//	RotateMatrix(&(**myAppData).fRotationMatrix, FixDiv(-90.0, 1), 0, 0);
		SetMovieMatrix(myMovie, &(**myAppData).fRotationMatrix);
	} else {
		return(false);
	}
	
	// get current port and GDevice
	GetGWorld(&mySavedPort, &mySavedGDevice);
	
	// get rid of any existing offscreen graphics world
	if ((**myAppData).fMovieGWorld != NULL)
		DisposeGWorld((**myAppData).fMovieGWorld);

	// clear out any existing custom cover function and reset the video media graphics mode
	// (these may have been modified for direct-screen drawing)
	SetMovieCoverProcs(myMovie, NULL, NULL, 0L);
	VRMoov_SetVideoGraphicsMode(myMovie, myAppData, false);
	
	// if necessary, create an offscreen graphics world;
	// this is where we'll image the movie before copying it into the back buffer
	// (which allows us to do special effects)
	if ((**myAppData).fUseMovieGWorld) {
		myErr = NewGWorld(&myGWorld, 0, &myRect, NULL, NULL, 0);
		if (myGWorld != NULL) 
			SetMovieGWorld(myMovie, myGWorld, GetGWorldDevice(myGWorld));
		(**myAppData).fMovieGWorld = myGWorld;
		(**myAppData).fMoviePixMap = GetGWorldPixMap(myGWorld);
		LockPixels((**myAppData).fMoviePixMap);
	} else {
		GetMovieGWorld((**theWindowObject).fMovie, &myGWorld, NULL);
		// set the video media graphics mode to drop out the chroma key color in a movie;
		// we also need to install an uncover function that doesn't erase the uncovered region
		if ((**myAppData).fCompositeMovie) {
			SetMovieCoverProcs(myMovie, NewMovieRgnCoverProc(VRMoov_CoverProc), NULL, (long)theWindowObject);
			VRMoov_SetVideoGraphicsMode(myMovie, myAppData, true);
		}
	}
	
	// install a back-buffer imaging procedure
	myInstance = (**theWindowObject).fInstance;
	if (myInstance != NULL)
		myErr = VRMoov_InstallBackBufferImagingProc(myInstance, theWindowObject);
		
	// start the movie playing in a loop
	VRMoov_LoopEmbeddedMovie(myMovie, myGWorld);
	
	return(true);
}
	
	
//////////
//
// VRMoov_LoopEmbeddedMovie
// Start the QuickTime movie playing in a loop.
//
//////////

void VRMoov_LoopEmbeddedMovie (Movie theMovie, GWorldPtr theGWorld)
{
	TimeBase		myTimeBase;

	SetMovieGWorld(theMovie, theGWorld, GetGWorldDevice(theGWorld));

	// throw the movie into loop mode
	myTimeBase = GetMovieTimeBase(theMovie);
	SetTimeBaseFlags(myTimeBase, GetTimeBaseFlags(myTimeBase) | loopTimeBase);

	// start playing the movie
	StartMovie(theMovie);
}
	

//////////
//
// VRMoov_DumpEmbeddedMovie
// Stop any existing embedded movie from playing and then clean up.
//
//////////

void VRMoov_DumpEmbeddedMovie (WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData;

	if (theWindowObject == NULL) 
		goto bail;
	
	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		goto bail;
	
	// if we have an embedded movie, stop it from playing and dispose of it
	if ((**myAppData).fMovie != NULL) {
		StopMovie((**myAppData).fMovie);
		DisposeMovie((**myAppData).fMovie);
		(**myAppData).fMovie = NULL;
	}
	
	// clear the existing back buffer imaging proc
	QTVRSetBackBufferImagingProc((**theWindowObject).fInstance, NULL, 0, NULL, 0);
	DisposeRoutineDescriptor((**myAppData).fBackBufferProc);
	(**myAppData).fBackBufferProc = NULL;
	
	// make sure the speaker isn't forced on
	//QTVRUtils_HideControllerButton((**theWindowObject).fController, kQTVRSpeakerButton);
	
	// make sure the back buffer is clean
	QTVRRefreshBackBuffer((**theWindowObject).fInstance, 0);

bail:
	return;
}
	

//////////
//
// VRMoov_InstallBackBufferImagingProc
// Install a back buffer imaging procedure.
// (This routine might sometimes be called to move or resize the area of interest within the panorama.)
//
//////////

OSErr VRMoov_InstallBackBufferImagingProc (QTVRInstance theInstance, WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData;
	OSErr					myErr = noErr;
	QTVRAreaOfInterest		myArea;
	float					myWidth, myHeight;

	if ((theInstance == NULL) || (theWindowObject == NULL)) 
		return(paramErr);

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL) 
		return(paramErr);

	HLock((Handle)myAppData);
	
	// remove any existing back buffer imaging procedure and dispose of its routine descriptor
	if ((**myAppData).fBackBufferProc != NULL) {
		QTVRSetBackBufferImagingProc(theInstance, NULL, 0, NULL, 0);
		DisposeRoutineDescriptor((**myAppData).fBackBufferProc);
		(**myAppData).fBackBufferProc = NULL;
	}

	// create a new routine descriptor
	(**myAppData).fBackBufferProc = NewQTVRBackBufferImagingProc(VRMoov_BackBufferImagingProc);
	
	// set the area of interest:
	// the application data structure holds the desired width, center, size, and scale of the movie
	myWidth = (**myAppData).fMovieWidth * (**myAppData).fMovieScale;
	myHeight = myWidth * (((float)(**myAppData).fMovieRect.right) / ((float)(**myAppData).fMovieRect.bottom));
	
	if ((**myAppData).fUseMovieCenter) {
		// use the stored movie center
		myArea.panAngle = (**myAppData).fMovieCenter.x + (myWidth/2);
		myArea.tiltAngle = (**myAppData).fMovieCenter.y + (myHeight/2);
	} else {
		// center the movie on the current pan and tilt angles
		myArea.panAngle = QTVRGetPanAngle(theInstance) + (myWidth/2);
		myArea.tiltAngle = QTVRGetTiltAngle(theInstance) + (myHeight/2);
	}
	
	myArea.width = myWidth;
	myArea.height = myHeight;
		
	// make sure we get called on every idle event, so we can keep playing the embedded movie;
	// also make sure we get called on every back buffer update
	if ((**myAppData).fCompositeMovie)
		myArea.flags = kQTVRBackBufferEveryIdle | kQTVRBackBufferEveryUpdate | kQTVRBackBufferAlwaysRefresh;
	else
		myArea.flags = kQTVRBackBufferEveryIdle | kQTVRBackBufferEveryUpdate;
	
	// install our procedure
	myErr = QTVRSetBackBufferImagingProc(theInstance, (**myAppData).fBackBufferProc, 1, &myArea, (SInt32)theWindowObject);

	HUnlock((Handle)myAppData);

	return(myErr);
}
	
	
//////////
//
// VRMoov_BackBufferImagingProc
// The back buffer imaging procedure: get a frame of movie and image it into the back buffer.
// Also, do any additional compositing that might be desired.
//
//////////

PASCAL_RTN OSErr VRMoov_BackBufferImagingProc (QTVRInstance theInstance, Rect *theRect, UInt16 theAreaIndex, UInt32 theFlagsIn, UInt32 *theFlagsOut, WindowObject theWindowObject)
{
#pragma unused(theAreaIndex)

	ApplicationDataHdl		myAppData;
	Movie					myMovie;
	Boolean					myIsDrawing = theFlagsIn & kQTVRBackBufferRectVisible;
	
	// assume we're not going to draw anything
	*theFlagsOut = 0;
	
	if ((theInstance == NULL) || (theWindowObject == NULL)) 
		return(paramErr);

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL) 
		return(paramErr);

	myMovie = (**myAppData).fMovie;
	if (myMovie != NULL) {
		// we have an embedded movie, so play it
		
		GWorldPtr			myGWorld, myMovGWorld;
		GDHandle			myGDevice, myMovGDevice;
		Rect				myRect;
		
		// reset movie matrix, if necessary
		if (!(**myAppData).fDrewOnPrevBBProc && myIsDrawing)
			SetMovieMatrix(myMovie, &(**myAppData).fRotationMatrix);

		// get the current graphics world
		// (on entry, the current graphics world is [usually] set to the back buffer)
		GetGWorld(&myGWorld, &myGDevice);

		// get the embedded movie's graphics world
		GetMovieGWorld(myMovie, &myMovGWorld, &myMovGDevice);

		// make sure that the movie GWorld is set correctly
		if ((**myAppData).fUseMovieGWorld) {
			// we're using an offscreen graphics world,
			// so set movie's GWorld to be that offscreen graphics world
			if (myMovGWorld != (**myAppData).fMovieGWorld)
				SetMovieGWorld(myMovie, (**myAppData).fMovieGWorld, GetGWorldDevice((**myAppData).fMovieGWorld));			
		} else {
			// we're not using an offscreen graphics world,
			// so set movie GWorld to be the back buffer;
			// (note that we call SetMovieGWorld only if we have to, for performance reasons)
			if ((myMovGWorld != myGWorld) || (myMovGDevice != myGDevice)) 
				SetMovieGWorld(myMovie, myGWorld, myGDevice);
		}
		
		if (myIsDrawing) {
			// make sure the movie's rect matches our area of interest
			GetMovieBox(myMovie, &myRect);
			if (!MacEqualRect(&myRect, theRect))
				SetMovieBox(myMovie, theRect);
			(**myAppData).fDrewOnPrevBBProc = true;
		} else {
			// if we're not visible, make sure we're not wasting time trying to draw
			MacSetRect(&myRect, 0, 0, 0, 0);
			SetMovieBox(myMovie, &myRect);
			(**myAppData).fDrewOnPrevBBProc = false;
		}
				
		// give the movie some time to play;
		// that is, draw a new frame into the movie's graphics world (and play movie sound)
		MoviesTask(myMovie, 0);
		
		// perform any additional compositing;
		// that is, draw, using the current chroma key color, anything to be dropped out of the image.
		// Note that this technique works *only* if we're using an offscreen graphics world.
		if ((**myAppData).fUseMovieGWorld && (**myAppData).fCompositeMovie && myIsDrawing) {
		
			RGBColor	myColor;
		
			// since we're using an offscreen graphics world, make sure we draw there
			SetGWorld((**myAppData).fMovieGWorld, GetGWorldDevice((**myAppData).fMovieGWorld));
	
			// set up compositing environment
			GetForeColor(&myColor);
			RGBForeColor(&(**myAppData).fChromaColor);
			
			// do the drawing
			if ((**myAppData).fHideRegion != NULL)
				MacPaintRgn((**myAppData).fHideRegion);
			
			// restore original drawing environment
			RGBForeColor(&myColor);
			
			// restore original graphics world
			SetGWorld(myGWorld, myGDevice);
		}
		
		if (myIsDrawing) {
		
			// if we're visible, say we're drawing
			*theFlagsOut = kQTVRBackBufferFlagDidDraw;
			
			// now, if we're using an offscreen graphics world, copy it into the back buffer
			if ((**myAppData).fUseMovieGWorld) {
				PixMapHandle	myPixMap;
				
				myPixMap = GetGWorldPixMap(myGWorld);
				LockPixels(myPixMap);
				
				// set the chroma key color, if necessary
				if ((**myAppData).fCompositeMovie)
					RGBBackColor(&(**myAppData).fChromaColor);
		
				// copy the current movie frame to the current graphics world
				CopyBits((BitMap *)(*(**myAppData).fMoviePixMap),
						 (BitMap *)(*myPixMap),
						  &(*(**myAppData).fMovieGWorld).portRect, 
						  theRect,
						  srcCopy | transparent, 
						  NULL);
					  
				// reset the chroma key color;
				// we need to do this because the buffer we just drew into might NOT actually
				// be the real back buffer (see Virtual Reality Programming With QuickTime VR, p. 1-154);
				// the copy between the intermediate buffer and the back buffer respects the current back color.
				if ((**myAppData).fCompositeMovie)
					RGBBackColor(&kWhiteColor);
					
				UnlockPixels(myPixMap);
			}
		}
		
	} else {
		// we don't have an embedded movie, so remove this back buffer imaging procedure
		VRMoov_DumpEmbeddedMovie(theWindowObject);
	}
	
	return(noErr);
}


//////////
//
// VRMoov_GetEmbeddedMovieWidth
// Get the width of the embedded movie.
//
//////////

float VRMoov_GetEmbeddedMovieWidth (WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData;
	float					myWidth;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		myWidth = 0;
	else
		myWidth = (**myAppData).fMovieWidth;
		
	return(myWidth);
}


//////////
//
// VRMoov_SetEmbeddedMovieWidth
// Set the width of the embedded movie.
//
//////////

void VRMoov_SetEmbeddedMovieWidth (WindowObject theWindowObject, float theWidth)
{
	ApplicationDataHdl		myAppData;
	QTVRInstance			myInstance;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;
		
	myInstance = (**theWindowObject).fInstance;
	if (myInstance == NULL)
		return;
	
	// install the desired width in our application data structure
	(**myAppData).fMovieWidth = theWidth;
	
	// clear out the existing area of interest
	QTVRRefreshBackBuffer(myInstance, 0);

	// reinstall the back buffer imaging procedure
	VRMoov_InstallBackBufferImagingProc(myInstance, theWindowObject);
}


//////////
//
// VRMoov_GetEmbeddedMovieCenter
// Get the center of the embedded movie.
//
//////////

void VRMoov_GetEmbeddedMovieCenter (WindowObject theWindowObject, QTVRFloatPoint *theCenter)
{
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL) {
		theCenter->x = 0.0;
		theCenter->y = 0.0;
	} else {
		theCenter->x = (**myAppData).fMovieCenter.x;
		theCenter->y = (**myAppData).fMovieCenter.y;
	}
		
}


//////////
//
// VRMoov_SetEmbeddedMovieCenter
// Set the center of the embedded movie.
//
//////////

void VRMoov_SetEmbeddedMovieCenter (WindowObject theWindowObject, const QTVRFloatPoint *theCenter)
{
	ApplicationDataHdl		myAppData;
	QTVRInstance			myInstance;
	float					myX, myY;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;
		
	myInstance = (**theWindowObject).fInstance;
	if (myInstance == NULL)
		return;
	
	myX = theCenter->x;
	myY = theCenter->y;
	
	// subject the values passed in to the current view constraints
	QTVRWrapAndConstrain(myInstance, kQTVRPan, myX, &myX);
	QTVRWrapAndConstrain(myInstance, kQTVRTilt, myY, &myY);
			
	// install the desired center in our application data structure
	(**myAppData).fMovieCenter.x = myX;
	(**myAppData).fMovieCenter.y = myY;
	
	// clear out the existing area of interest
	QTVRRefreshBackBuffer(myInstance, 0);

	// reinstall the back buffer imaging procedure
	VRMoov_InstallBackBufferImagingProc(myInstance, theWindowObject);
}


//////////
//
// VRMoov_GetEmbeddedMovieScale
// Get the scale of the embedded movie.
//
//////////

float VRMoov_GetEmbeddedMovieScale (WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData;
	float					myScale;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		myScale = 0;
	else
		myScale = (**myAppData).fMovieScale;
		
	return(myScale);
}


//////////
//
// VRMoov_SetEmbeddedMovieScale
// Set the scale factor of the embedded movie.
//
//////////

void VRMoov_SetEmbeddedMovieScale (WindowObject theWindowObject, float theScale)
{
	ApplicationDataHdl		myAppData;
	QTVRInstance			myInstance;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;
		
	myInstance = (**theWindowObject).fInstance;
	if (myInstance == NULL)
		return;
	
	// install the desired scale factor in our application data structure
	(**myAppData).fMovieScale = theScale;
	
	// clear out the existing area of interest
	QTVRRefreshBackBuffer(myInstance, 0);

	// reinstall the back buffer imaging procedure
	VRMoov_InstallBackBufferImagingProc(myInstance, theWindowObject);
}


//////////
//
// VRMoov_GetEmbeddedMovieRect
// Get the rectangle of the embedded movie.
//
//////////

void VRMoov_GetEmbeddedMovieRect (WindowObject theWindowObject, Rect *theRect)
{
	ApplicationDataHdl		myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL) {
		theRect->left = 0.0;
		theRect->top = 0.0;
		theRect->right = 0.0;
		theRect->bottom = 0.0;
	} else {
		theRect->left = (**myAppData).fMovieRect.left;
		theRect->top = (**myAppData).fMovieRect.top;
		theRect->right = (**myAppData).fMovieRect.right;
		theRect->bottom = (**myAppData).fMovieRect.bottom;
	}
		
}


//////////
//
// VRMoov_SetEmbeddedMovieRect
// Set the rectangle of the embedded movie.
//
//////////

void VRMoov_SetEmbeddedMovieRect (WindowObject theWindowObject, const Rect *theRect)
{
	ApplicationDataHdl		myAppData;
	QTVRInstance			myInstance;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;
		
	myInstance = (**theWindowObject).fInstance;
	if (myInstance == NULL)
		return;
	
	// install the desired rectangle in our application data structure
	(**myAppData).fMovieRect.right = theRect->right;
	(**myAppData).fMovieRect.bottom = theRect->bottom;

	// clear out the existing area of interest
	QTVRRefreshBackBuffer(myInstance, 0);

	// reinstall the back buffer imaging procedure
	VRMoov_InstallBackBufferImagingProc(myInstance, theWindowObject);
}
	

//////////
//
// VRMoov_SetChromaColor
// Set the chroma key color for a window object:
// display color picker dialog and remember the newly-selected color.
//
//////////

void VRMoov_SetChromaColor (WindowObject theWindowObject)
{
#if TARGET_OS_MAC
	ColorPickerInfo		theInfo;
	ApplicationDataHdl	myAppData;

	myAppData = (ApplicationDataHdl)GetAppDataFromWindowObject(theWindowObject);
	if (myAppData == NULL)
		return;

	// pass in existing color
	theInfo.theColor.color.rgb.red = (**myAppData).fChromaColor.red;
	theInfo.theColor.color.rgb.green = (**myAppData).fChromaColor.green;
	theInfo.theColor.color.rgb.blue = (**myAppData).fChromaColor.blue;
	
	// not much here...
	theInfo.theColor.profile = 0L;
	theInfo.dstProfile = 0L;
	theInfo.flags = 0L;
	theInfo.placeWhere = kCenterOnMainScreen;
	theInfo.pickerType = 0L;
	theInfo.eventProc = gColorFilterUPP;
	theInfo.colorProc = NULL;
	theInfo.colorProcData = 0L;
	GetIndString(theInfo.prompt, kColorPickerTextStringID, 1);
	
	// set Edit menu info
	theInfo.mInfo.editMenuID = mEdit;
	theInfo.mInfo.undoItem = iUndo;
	theInfo.mInfo.cutItem = iCut;
	theInfo.mInfo.copyItem = iCopy;
	theInfo.mInfo.pasteItem = iPaste;
	theInfo.mInfo.clearItem = iClear;
	
	// call Color Picker
	if (PickColor(&theInfo) == noErr && theInfo.newColorChosen) {
		// install the newly chosen color in the palette
		(**myAppData).fChromaColor.red = theInfo.theColor.color.rgb.red;
		(**myAppData).fChromaColor.green = theInfo.theColor.color.rgb.green;
		(**myAppData).fChromaColor.blue = theInfo.theColor.color.rgb.blue;
	}
#endif
}
	

//////////
//
// VRMoov_ColorDialogEventFilter
// This is our color dialog event filter. 
//
//////////

PASCAL_RTN Boolean VRMoov_ColorDialogEventFilter (EventRecord *theEvent)
{
	OSErr			theErr = noErr;
	Boolean			theEventHandled = false;

#if TARGET_OS_MAC
	switch (theEvent->what) {
		case updateEvt: {
			if ((WindowPtr)theEvent->message != FrontWindow()) {
				DoUpdateWindow((WindowPtr)theEvent->message, &((WindowPtr)theEvent->message)->portRect);
				theEventHandled = true;
			}
			break;
		}

		case kHighLevelEvent:
			theErr = AEProcessAppleEvent(theEvent);
			if (!theErr)
				theEventHandled = true;
		 	break;
			
		default:
			theEventHandled = false;
			break;
	}	
	
#endif
	return(theEventHandled);
}


//////////
//
// VRMoov_CoverProc
// The cover function of the embedded movie.
//
//////////

PASCAL_RTN OSErr VRMoov_CoverProc (Movie theMovie, RgnHandle theRegion, WindowObject theWindowObject)
{
	return(noErr);
}


//////////
//
// VRMoov_SetVideoGraphicsMode
// Set the video media graphics mode of the embedded movie.
//
//////////

void VRMoov_SetVideoGraphicsMode (Movie theMovie, ApplicationDataHdl theAppData, Boolean theSetVGM)
{
	long				myTrackCount;
	short				myIndex;
	Track				myMovieTrack = NULL;
	Media				myMedia;
	OSType				myMediaType;
	
	myTrackCount = GetMovieTrackCount(theMovie);
	for (myIndex = 1; myIndex <= myTrackCount; myIndex++) {
		myMovieTrack = GetMovieIndTrack(theMovie, myIndex);
		myMedia = GetTrackMedia(myMovieTrack);
		GetMediaHandlerDescription(myMedia, &myMediaType, NULL, NULL);
		if (myMediaType == VideoMediaType) {
			if (theSetVGM)
				MediaSetGraphicsMode(GetMediaHandler(myMedia), srcCopy | transparent, &(**theAppData).fChromaColor);
			else
				MediaSetGraphicsMode(GetMediaHandler(myMedia), srcCopy, &kWhiteColor);
		}
	}
	
	return;
}


//////////
//
// VRMoov_GetVideoGraphicsPixelDepth
// Return the highest pixel depth supported by a QuickTime movie.
//
//////////

short VRMoov_GetVideoGraphicsPixelDepth (Movie theMovie)
{
	long				myTrackCount;
	short				myIndex;
	Track				myMovieTrack = NULL;
	Media				myMedia;
	OSType				myMediaType;
	short				myQuality;
	
	myTrackCount = GetMovieTrackCount(theMovie);
	for (myIndex = 1; myIndex <= myTrackCount; myIndex++) {
		myMovieTrack = GetMovieIndTrack(theMovie, myIndex);
		myMedia = GetTrackMedia(myMovieTrack);
		GetMediaHandlerDescription(myMedia, &myMediaType, NULL, NULL);
		if (myMediaType == VideoMediaType) {
			myQuality = GetMediaQuality(myMedia);
			if (myQuality >> 5)
				return(32);
			if (myQuality >> 4)
				return(16);
			if (myQuality >> 3)
				return(8);
			if (myQuality >> 2)
				return(4);
			if (myQuality >> 1)
				return(2);
			if (myQuality >> 0)
				return(1);
		}
	}
	
	return(0);
}


