/*
	File:		Application.c

	Contains:	QuickTime Streaming sample application.

	Written by:	Scott Kuechle

	Copyright:	 1999 by Apple Computer, Inc., all rights reserved.

	Change History (most recent first):

	   <1>	 	11/20/98		srk		first file;
	    
 
 NOTES:

 
 TO DO:

*/

/************************************************************
*                                                           *
*    INCLUDE FILES                                          *
*                                                           *
*************************************************************/

#include "Application.h"
#include "QTCode.h"
#include "Queue.h"

/************************************************************
*                                                           *
*    FUNCTION PROTOTYPES                                    *
*                                                           *
*************************************************************/

long FAR PASCAL		FrameWndProc  (HWND, UINT, UINT, LONG) ;
long FAR PASCAL		MovieWndProc  (HWND, UINT, UINT, LONG) ;
static int			DoIdleMenus(HWND hWnd, HMENU hMenu);
static void			DoCut(HWND);
static void			DoCopy(HWND);
static void			DoPaste(HWND);
static void			DoClear(HWND);
static void			DoUndo(HWND);
static void			DoAbout(void);
static void			DoEraseRect(HDC	hDC, RECT *theRect);

static HWND			CreateChildWindow(HWND ownerWindow, HINSTANCE appInstance);
static void			SetWindowTitle(HWND hwnd);
static BOOL			DoGetFile(HWND hwndOwner, char *fileBuffer, DWORD bufferSize);
static BOOL			GetWinPrivateStorage(HWND hwnd, LPVOID privateStorage, int storageSize);
static void			CheckForNewPresentations(void);
static void			ShowTheWindow(HWND hwnd);
static void			SetWinPrivateStorage(HWND hwnd, LPVOID privateStorage, int storageSize);



/************************************************************
*                                                           *
*    GLOBAL VARIABLES                                       *
*                                                           *
*************************************************************/

static	HANDLE			ghInst					= NULL;
static	HWND			ghWndFrame				= NULL;
static	HWND			ghWndMDIClient			= NULL;
static	BOOL			gWeAreSizingWindow		= 0;
static	BOOL			gWeAreCreatingWindow	= 0;
static	short			gOpenMovieCount			= 0;
static	char			gAppName[20];
static	char			gChildName[20];

static	int				gReceiveWindowCount		= 0;
 
/************************************************************
*                                                           *
*    CONSTANTS                                              *
*                                                           *
*************************************************************/

#define		kMillsPerTick			10			/* milliseconds per timer tick */
#define		kReceiveWindowTitle		"Receive "

#define		BailNil(n)		if (!n) goto bail;
#define		BailError(n)	if (n) goto bail;





/* ---------------------------------------------------------------- */
/*                                                                  */
/*    WinMain()                                                     */
/*                                                                  */
/*                                                                  */
/*    The main function for this application                        */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
					LPSTR lpszCmdLine, int nCmdShow)
    {
	HANDLE		hAccel ;
	HWND		hwndFrame;
	MSG			msg ;
    WNDCLASSEX	wc;
	OSErr		err;


		ghInst = hInstance ;

			/* Initialize QuickTime */
		err = QTCode_DoQTInit();
		if (err != noErr)
		{
			MessageBox(hwndFrame, "QuickTime Streaming Initialization failure", "", MB_OK);
			return msg.wParam;
		}


		Queue_InitQHdr_();
		Queue_InitQ_();

		if (!hPrevInstance) 
		{
			LoadString(hInstance, IDS_APPNAME, gAppName, sizeof(gAppName));
			
			// Register the frame window class
			wc.cbSize        = sizeof(WNDCLASSEX);
			wc.style         = CS_HREDRAW | CS_VREDRAW;
			wc.lpfnWndProc   = (WNDPROC)FrameWndProc;
			wc.cbClsExtra    = 0;
			wc.cbWndExtra    = 0;
			wc.hInstance     = hInstance;
			wc.hIcon         = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_APPICON));
			wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
			wc.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
			wc.lpszMenuName  = gAppName;
			wc.lpszClassName = gAppName;
			wc.hIconSm       = LoadImage(hInstance,
										 MAKEINTRESOURCE(IDI_APPICON),
										 IMAGE_ICON,
										 16, 16,
										 0);
			if (!RegisterClassEx(&wc))
			{
				if (!RegisterClass((LPWNDCLASS)&wc.style))
        			return FALSE;
			}

			LoadString(hInstance, IDS_MDICHILD, gChildName, sizeof(gChildName));

			// Register the Movie child window class
			wc.cbSize        = sizeof(WNDCLASSEX);
			wc.style         = CS_HREDRAW | CS_VREDRAW;
			wc.lpfnWndProc   = (WNDPROC)MovieWndProc;
			wc.cbClsExtra    = 0;
			wc.cbWndExtra    = 0;
			wc.hInstance     = hInstance;
			wc.hIcon         = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_CHILDICON));
			wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
			wc.hbrBackground = NULL	/* we want to handle erasing of the background ourselves */;
			wc.lpszMenuName  = NULL;
			wc.lpszClassName = gChildName;
			wc.hIconSm       = LoadImage(hInstance,
										 MAKEINTRESOURCE(IDI_CHILDICON),
										 IMAGE_ICON,
										 16, 16,
										 0);
			if (!RegisterClassEx(&wc))
			{
				if (!RegisterClass((LPWNDCLASS)&wc.style))
        			return FALSE;
			}
		}

		// Load accelerators
		hAccel = LoadAccelerators (hInstance, gAppName);

		// Create the main frame window
		ghWndFrame = hwndFrame = CreateWindow (gAppName, gAppName,
											   WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
											   CW_USEDEFAULT, CW_USEDEFAULT,
											   CW_USEDEFAULT, CW_USEDEFAULT,
											   NULL, NULL, hInstance, NULL) ;
 
			/* Show the window */
		ShowWindow(hwndFrame, nCmdShow);
		UpdateWindow(hwndFrame);

			/* Process messages */
		while (GetMessage(&msg, NULL, 0, 0))
		{
			if (!TranslateMDISysAccel(ghWndMDIClient, &msg))
				if (!TranslateAccelerator(hwndFrame, hAccel, &msg))
				{
					TranslateMessage(&msg);
					DispatchMessage(&msg);
				}
			/* here's where we check for any
				new presentations - if there
				are any, we'll create a new
				window for it */
			CheckForNewPresentations();
		}

			/* QuickTime clean up */
		QTCode_QTCleanUp();

		return msg.wParam;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    CheckForNewPresentations()                                    */
/*                                                                  */
/*                                                                  */
/*    Check our presentation queue for new entries. If one is found */
/*    we will create a new window for the presentation.             */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

void CheckForNewPresentations(void)
{
	MyQElemRecPtr	qElemPtr = NULL;
		
			/* Are there any new presentations? */
		qElemPtr = Queue_GetUsedQElem();
		if (qElemPtr != NULL)
		{
			ChildWindowRecord	childWinRec;
			HWND				newWindow = NULL;
			BOOL				success = false;
			OSErr				err;

					/* got a new presentation! so let's create a new window for it */
				newWindow = CreateChildWindow(ghWndMDIClient, ghInst);
				BailNil(newWindow);

					/* get the ChildWindowRecord associated with this window */
				success = GetWinPrivateStorage(newWindow,
											(LPVOID) &childWinRec,
											sizeof (struct ChildWindowRecord));

				SetWindowTitle(newWindow);

					/* save off the presentation value in the window's
						private storage */
				childWinRec.aPresentation = Queue_GetQElemPresValue(qElemPtr);

				err = QTCode_ConfigurePresentation(childWinRec.aPresentation,
													kQTSAllStreams,
													childWinRec.grafPort);
				BailError(err);

					/* save the ChildWindowRecord with the window */
				SetWinPrivateStorage(newWindow,
									(LPVOID) &childWinRec,
									sizeof (struct ChildWindowRecord));

				ShowTheWindow(newWindow);

					/* put this queue element back into the "free" queue */
				Queue_PutElemIntoFreeQueue(qElemPtr);
			bail:
				;

		}

}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    FrameWndProc()                                                */
/*                                                                  */
/*                                                                  */
/*    The window procedure for the MDI frame window                 */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

long FAR PASCAL FrameWndProc (HWND hwnd, UINT message, UINT wParam, LONG lParam)
{
	HWND               hwndChild ;

	switch (message)
	{
		case WM_CREATE:          // Create the client window
		{
			CLIENTCREATESTRUCT ccs = {0};

			ccs.hWindowMenu  = GetSubMenu(GetMenu(hwnd), WINDOWMENU);
			ccs.idFirstChild = IDM_WINDOWCHILD;

			// Create the MDI client filling the client area
			ghWndMDIClient = CreateWindow("mdiclient",
										 NULL,
										 WS_CHILD | WS_CLIPCHILDREN | WS_VSCROLL |
										 WS_HSCROLL,
										 0, 0, 0, 0,
										 hwnd,
										 (HMENU)0xCAC,
										 ghInst,
										 (LPVOID)&ccs);

			ShowWindow(ghWndMDIClient, SW_SHOW);
		}
		return 0 ;

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case IDM_FILEOPEN:
					{
						char	szPathName[256];

							/* get pointer to sdp file from user */
						if (DoGetFile(ghWndMDIClient, szPathName, sizeof(szPathName)))
						{
							OSErr				err;
							QTSPresentation		aPresentation;

									/* create a new presentation using the sdp file selected by the user */
								err = QTCode_NewPresFromFile(szPathName,		/* sdp file */
															 &aPresentation);	/* presentation value returned here */
						}
					}

					return 0 ;

				case IDM_FILECLOSE:
					hwndChild = (HWND)SendMessage(ghWndMDIClient, WM_MDIGETACTIVE, 0, 0L) ;
					if (SendMessage (hwndChild, WM_QUERYENDSESSION, 0, 0L))
					{
						SendMessage (ghWndMDIClient, WM_MDIDESTROY, (WPARAM)hwndChild, 0L) ;
					}
					return 0 ;

				case IDM_EXIT:
					SendMessage (hwnd, WM_CLOSE, 0, 0L) ;
					return 0 ;

				case IDM_WINDOWTILE:
					SendMessage (ghWndMDIClient, WM_MDITILE, 0, 0L) ;
					return 0 ;

				case IDM_WINDOWCASCADE:
					SendMessage (ghWndMDIClient, WM_MDICASCADE, 0, 0L) ;
					return 0 ;

				case IDM_WINDOWICONS:
					SendMessage (ghWndMDIClient, WM_MDIICONARRANGE, 0, 0L) ;
					return 0 ;

				case IDM_WINDOWCLOSEALL:
					{
						HWND	hwndT;
			
						while (hwndT = GetWindow(ghWndMDIClient, GW_CHILD))
						{
							// Skip the icon and title windows
							while (hwndT && GetWindow(hwndT, GW_OWNER))
								hwndT = GetWindow(hwndT, GW_HWNDNEXT);

							if (!hwndT) break;

							SendMessage(ghWndMDIClient, WM_MDIDESTROY, (WPARAM)hwndT, 0L);
						}
					}
					return 0;

				case IDM_ABOUT:
					DoAbout();
					return 0;

				default:            // Pass to active child
					hwndChild = (HWND)SendMessage (ghWndMDIClient, WM_MDIGETACTIVE, 0, 0L) ;
					if (IsWindow (ghWndMDIClient))
						SendMessage (hwndChild, WM_COMMAND, wParam, lParam) ;

					break ;        // and then to DefFrameProc
			}
			break ;

		case WM_INITMENU:
			if (GetMenu(hwnd) == (HMENU)wParam)
				return DoIdleMenus((HWND)SendMessage(ghWndMDIClient, WM_MDIGETACTIVE, 0, 0), (HMENU)wParam);
			return 1;

		case WM_DESTROY :
			PostQuitMessage (0) ;
			return 0 ;
	}

	return DefFrameProc (hwnd, ghWndMDIClient, message, wParam, lParam) ;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    MovieWndProc()                                                */
/*                                                                  */
/*                                                                  */
/*    The window procedure for our movie window                     */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

long FAR PASCAL MovieWndProc (HWND hwnd, UINT message, UINT wParam, LONG lParam)
{
	HGLOBAL			hWinStorage = NULL;
	ChildWindowPtr	childWinPtr = NULL;
	MSG				msg = {0};


	hWinStorage = (HGLOBAL)GetWindowLong(hwnd, GWL_USERDATA);

	switch (message)
	{
		case WM_CREATE:
			{
				CGrafPtr gp;

						/* Tuck away some private storage */
					hWinStorage = GlobalAlloc(GMEM_MOVEABLE + GMEM_ZEROINIT, sizeof(ChildWindowRecord));
					
						/* Associate a GrafPort with this window */
					gp = (CGrafPtr)QTCode_DoCreatePortAssociation(hwnd, NULL, 0);

					childWinPtr = (ChildWindowPtr)GlobalLock(hWinStorage);
					if (childWinPtr != NULL)
					{
						childWinPtr->grafPort = gp;
						childWinPtr->windowTimer = SetTimer(hwnd,
															0,
															kMillsPerTick,
															NULL			/* use WM_TIMER messages */
															);
					}
					SetWindowLong(hwnd, GWL_USERDATA, (LPARAM)hWinStorage);
			}
			return 0;

		case WM_GETMINMAXINFO:
			{
				if (hWinStorage != NULL)
				{
					childWinPtr = (ChildWindowPtr)GlobalLock(hWinStorage);
					if (childWinPtr != NULL)
					{
						LPMINMAXINFO lpmmi;
						BOOL success = false;
						RECT rect;
						LONG width, height;

							/* build a rectangle for client area of the desired size */
							rect.left	= 0;
							rect.top	= 0;
							rect.bottom = kPresHeight;
							rect.right	= kPresWidth;

							/* build a client rect which will accomodate the menu bar, border, etc. for
								a given window type */

							success = AdjustWindowRectEx(&rect, WS_OVERLAPPEDWINDOW, FALSE, WS_EX_MDICHILD);
							width = (rect.right) - (rect.left);
							height = (rect.bottom) - (rect.top);

							lpmmi = (LPMINMAXINFO) lParam;

							lpmmi->ptMaxSize.x = width;
							lpmmi->ptMaxSize.y = height;
					
							lpmmi->ptMaxPosition.x = width;
							lpmmi->ptMaxPosition.y = height;

							lpmmi->ptMinTrackSize.x = width;
							lpmmi->ptMinTrackSize.y = height;

							lpmmi->ptMaxTrackSize.x = width;
							lpmmi->ptMaxTrackSize.y = height;

							GlobalUnlock(hWinStorage);
					}
				}
			}

			return 0;

		case WM_PAINT:
			{
				PAINTSTRUCT	lPaint;
				HDC			hDC;

					hDC = BeginPaint(hwnd, &lPaint);
					if (hDC != NULL)
					{
						if (hWinStorage != NULL)
						{
							childWinPtr = (ChildWindowPtr)GlobalLock(hWinStorage);
							if (childWinPtr != NULL)
							{
								if (childWinPtr->aPresentation != NULL)
								{
									QTCode_UpdatePresentation(childWinPtr->aPresentation);
								}
								GlobalUnlock(hWinStorage);
							}
						}
					}
					EndPaint(hwnd, &lPaint);
			}
			return 0;


		case WM_TIMER:
			{
				if (hWinStorage != NULL)
				{
					childWinPtr = (ChildWindowPtr)GlobalLock(hWinStorage);
					if (childWinPtr != NULL)
					{
						if (childWinPtr->aPresentation != NULL)
						{
								/* give idle time to the presentation */
							QTSPresIdle(childWinPtr->aPresentation, nil);
						}
						GlobalUnlock(hWinStorage);
					}
				}

			}
			return 0;

		case WM_COMMAND:
			{
				switch(LOWORD(wParam))		// Undo, Cut, Copy, Paste and Clear
				{
					case IDM_EDITUNDO:
						DoUndo(hwnd);
						break;

					case IDM_EDITCUT:
						DoCut(hwnd);
						break;

					case IDM_EDITCOPY:
						DoCopy(hwnd);
						break;

					case IDM_EDITPASTE:
						DoPaste(hwnd);
						break;

					case IDM_EDITCLEAR:
						DoClear(hwnd);
						break;
				}
			}
			break;

		case WM_DESTROY:
			{		
				if (hWinStorage != NULL)
				{
					childWinPtr = (ChildWindowPtr)GlobalLock(hWinStorage);
					if (childWinPtr != NULL)
					{
							/* One less movie open */
						--gOpenMovieCount;
						
						KillTimer( hwnd, childWinPtr->windowTimer );

						if (childWinPtr->aPresentation != NULL)
						{
								/* get rid of this presentation */
							QTCode_DestroyPresentation(childWinPtr->aPresentation,
														kQTSAllStreams);
						}
							/* Destroy the port association */
						QTCode_DoDestroyPortAssociation(hwnd);

						SetWindowLong(hwnd, GWL_USERDATA, 0);
						GlobalUnlock(hWinStorage);
						GlobalFree((HGLOBAL)GetWindowLong(hwnd, GWL_USERDATA));
					}
				}
			}
			return 0;
	}

	return DefMDIChildProc (hwnd, message, wParam, lParam);
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    CreateChildWindow()                                           */
/*                                                                  */
/*                                                                  */
/*    Code to open a given movie and prepare it for display in a    */
/*    window                                                        */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

HWND CreateChildWindow(HWND ownerWindow, HINSTANCE appInstance)
{
    DWORD			dwVersion;
    HWND			hwndChild = 0;
	char            szFileName[256];
	BOOL			success = false;
	RECT			rect;
	LONG			width, height;

		/* build a rectangle for client area of the desired size */
		rect.left	= 0;
		rect.top	= 0;
		rect.bottom = kPresHeight;
		rect.right	= kPresWidth;

		/* build a client rect which will accomodate the menu bar, border, etc. for
			a given window type */
		success = AdjustWindowRectEx(&rect, WS_OVERLAPPEDWINDOW, FALSE, WS_EX_MDICHILD);

		width = (rect.right) - (rect.left);
		height = (rect.bottom) - (rect.top);

			// Create the child movie window
		dwVersion = GetVersion();
		gWeAreCreatingWindow = TRUE;

		if ((dwVersion < 0x80000000) || (LOBYTE(LOWORD(dwVersion)) < 4))
		{
			// This is Windows NT or Win32s, so use the WM_MDICREATE message
			MDICREATESTRUCT mcs;

			mcs.szClass = gChildName;
			mcs.szTitle = szFileName; 
			mcs.hOwner  = appInstance;
			mcs.x       = CW_USEDEFAULT;
			mcs.y       = CW_USEDEFAULT;
			mcs.cx      = width;	/* set an appropriate width for our movie window */
			mcs.cy      = height;	/* set an appropriate height for our movie window */
			mcs.style   = 0;
			mcs.lParam  = 0;

			hwndChild = (HWND) SendMessage(ownerWindow,
										   WM_MDICREATE,
										   0,
										   (LPARAM)(LPMDICREATESTRUCT) &mcs);
		}
		else
		{
			// This method will only work with Windows 95, not Windows NT or Win32s
			hwndChild = CreateWindowEx(WS_EX_MDICHILD,
									   gChildName,
									   szFileName,
									   0,
									   CW_USEDEFAULT,
									   CW_USEDEFAULT,
									   width,	/* set an appropriate width for our movie window */
									   height,	/* set an appropriate height for our movie window */
									   ownerWindow, 
									   NULL,
									   appInstance,
									   0);
		}
		gWeAreCreatingWindow = FALSE;

		return hwndChild;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    SetWindowTitle()                                              */
/*                                                                  */
/*                                                                  */
/*    Sets the title of a presentation window                       */
/*                                                                  */
/* ---------------------------------------------------------------- */

void SetWindowTitle(HWND hwnd)
{
    char	winName[256];
	Str255	winCount;

		if (hwnd)
		{
				/* count of number of receive windows */
			++gReceiveWindowCount;	
			strcpy(winName, kReceiveWindowTitle);
			NumToString(gReceiveWindowCount, winCount);

			strcat(winName, p2cstr(winCount));
			SetWindowText(hwnd, winName);
		}
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    ShowTheWindow()                                               */
/*                                                                  */
/*                                                                  */
/*    Code to display a window                                      */
/*                                                                  */
/* ---------------------------------------------------------------- */

void ShowTheWindow(HWND hwnd)
{
	// Show the window
	ShowWindow(hwnd, SW_SHOW);
	UpdateWindow(hwnd);
	SetFocus(hwnd);

	// One more window has been opened
	gOpenMovieCount += 1;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    GetWinPrivateStorage()                                        */
/*                                                                  */
/*                                                                  */
/*    Code to get the private storage for a window                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

BOOL GetWinPrivateStorage(HWND hwnd, LPVOID privateStorage, int storageSize)
{
	HGLOBAL			hWinStorage = NULL;

			// Store movie info in private window record
		hWinStorage = (HGLOBAL)GetWindowLong(hwnd, GWL_USERDATA);
		if (hWinStorage != NULL)
		{

			LPVOID	storagePtr = NULL;

			storagePtr = GlobalLock(hWinStorage);
			if (storagePtr != NULL)
			{
				memcpy(privateStorage, storagePtr, storageSize);
				GlobalUnlock(hWinStorage);

				return true;
			}
		}

		return false;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    SetWinPrivateStorage()                                        */
/*                                                                  */
/*                                                                  */
/*    Code to set the private storage for a window                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

void SetWinPrivateStorage(HWND hwnd, LPVOID privateStorage, int storageSize)
{
	HGLOBAL			hWinStorage = NULL;
			
			// Store movie info in private window record
		hWinStorage = (HGLOBAL)GetWindowLong(hwnd, GWL_USERDATA);
		if (hWinStorage != NULL)
		{
			LPVOID	storagePtr = NULL;

			storagePtr = GlobalLock(hWinStorage);
			if (storagePtr != NULL)
			{
				memcpy(storagePtr, privateStorage, storageSize);
				GlobalUnlock(hWinStorage);
			}
		}
		SetWindowLong(hwnd, GWL_USERDATA, (LPARAM)hWinStorage);

}


/* ---------------------------------------------------------------- */
/*                                                                  */
/*    DoGetFile()                                                   */
/*                                                                  */
/*                                                                  */
/*    Present the user with the standard open file dialog           */
/*                                                                  */
/* ---------------------------------------------------------------- */
BOOL DoGetFile(HWND hwndOwner, char *fileBuffer, DWORD bufferSize)
{
    OPENFILENAME    ofn = {0};

			// Prompt the user for an SDP file...
		*fileBuffer = 0;

		ofn.lStructSize		= sizeof(OPENFILENAME);
		ofn.hwndOwner		= hwndOwner;
		ofn.lpstrFile		= (LPSTR)fileBuffer;
		ofn.nMaxFile		= bufferSize;
		ofn.lpstrFilter		= "Open an SDP file (*.sdp) \0 *.sdp\0All Files (*.*) \0 *.*\0";
		ofn.nFilterIndex	= 1;
		ofn.lpstrInitialDir = NULL;
		ofn.Flags			= OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

		return(GetOpenFileName(&ofn));

}


/* ---------------------------------------------------------------- */
/*                                                                  */
/*    DoCut()                                                       */
/*                                                                  */
/*                                                                  */
/*    Handles clipboard "cut" operations                            */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static void DoCut(HWND hwnd)
{
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    DoCopy()                                                      */
/*                                                                  */
/*                                                                  */
/*    Handles clipboard "copy" operations                           */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static void DoCopy(HWND hwnd)
{
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    DoPaste()                                                     */
/*                                                                  */
/*                                                                  */
/*    Handles clipboard "paste" operations                          */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static void DoPaste(HWND hwnd)
{
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    DoClear()                                                     */
/*                                                                  */
/*                                                                  */
/*    Handles clipboard "clear" operations                          */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static void DoClear(HWND hwnd)
{
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    DoUndo()                                                      */
/*                                                                  */
/*                                                                  */
/*    Handles clipboard "undo" operations                           */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static void DoUndo(HWND hwnd)
{
}


/* ---------------------------------------------------------------- */
/*                                                                  */
/*    DoIdleMenus()                                                 */
/*                                                                  */
/*                                                                  */
/*    Enable the close item if there are any movie windows opened   */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static int DoIdleMenus(HWND hWnd, HMENU hMenu)
{
		/* Enable the close item if there are any movie windows opened */
	EnableMenuItem(hMenu, IDM_FILECLOSE, (gOpenMovieCount) ? MF_ENABLED : MF_GRAYED);

	return 0;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    DialogProc()                                                  */
/*                                                                  */
/*                                                                  */
/*    Callback for our dialog box                                   */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static LRESULT CALLBACK DialogProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case IDOK:
					EndDialog(hDlg, IDOK);
					break;
			}
			break;
	}

    return 0;
}

/* ---------------------------------------------------------------- */
/*                                                                  */
/*    DoAbout()                                                     */
/*                                                                  */
/*                                                                  */
/*    Displays a dialog box with information about our program      */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

static void DoAbout()
{
	DialogBox(ghInst, MAKEINTRESOURCE(IDD_ABOUT), ghWndFrame, (DLGPROC)DialogProc);
}




/* ---------------------------------------------------------------- */
/*                                                                  */
/*    DoEraseRect()                                                 */
/*                                                                  */
/*                                                                  */
/*    Code to erase a given windows rectangle                       */
/*                                                                  */
/*                                                                  */
/*                                                                  */
/* ---------------------------------------------------------------- */

void DoEraseRect(HDC	hDC,
				RECT	*theRect)
{
	if (hDC != NULL)
	{
		HBRUSH		aBrush;
			
				/* erase rect. for string before redraw */
			aBrush = CreateSolidBrush(RGB(255,255,255));
			if (aBrush != NULL)
			{
				FillRect(hDC, theRect, aBrush);
				DeleteObject(aBrush);
			}

	}
}
